% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/NIPALS.pls.R
\name{NIPALS.pls}
\alias{NIPALS.pls}
\title{Partial Least Squares Regression via NIPALS (Internal)}
\usage{
NIPALS.pls(x, y, n.components = NULL)
}
\arguments{
\item{x}{A numeric matrix or data frame of predictors (X). Should have dimensions n × p.}

\item{y}{A numeric matrix or data frame of response variables (Y). Should have dimensions n × q.}

\item{n.components}{Integer specifying the number of PLS components to extract. If NULL, it defaults to \code{qr(x)$rank}.}
}
\value{
A list with the following elements:
\describe{
  \item{model.type}{Character string indicating the model type ("PLS Regression").}
  \item{T}{Matrix of X scores (n × H).}
  \item{U}{Matrix of Y scores (n × H).}
  \item{W}{Matrix of X weights (p × H).}
  \item{C}{Matrix of normalized Y weights (q × H).}
  \item{P_loadings}{Matrix of X loadings (p × H).}
  \item{Q_loadings}{Matrix of Y loadings (q × H).}
  \item{B_vector}{Vector of regression scalars (length H), one for each component.}
  \item{coefficients}{Matrix of regression coefficients in original data scale (p × q).}
  \item{intercept}{Vector of intercepts (length q). Always zero here due to centering.}
  \item{X_explained}{Percent of total X variance explained by each component.}
  \item{Y_explained}{Percent of total Y variance explained by each component.}
  \item{X_cum_explained}{Cumulative X variance explained.}
  \item{Y_cum_explained}{Cumulative Y variance explained.}
}
}
\description{
This function is called internally by \code{\link{pls.regression}} and is not intended
to be used directly. Use \code{pls.regression(..., calc.method = "NIPALS")} instead.

Performs Partial Least Squares (PLS) regression using the NIPALS (Nonlinear Iterative Partial Least Squares) algorithm.
This method estimates the latent components (scores, loadings, weights) by iteratively updating the X and Y score directions
until convergence. It is suitable for cases where the number of predictors is large or predictors are highly collinear.
}
\details{
The algorithm standardizes both \code{x} and \code{y} using z-score normalization. It then performs the following for each
of the \code{n.components} latent variables:
\enumerate{
  \item Initializes a random response score vector \eqn{u}.
  \item Iteratively:
    \itemize{
      \item Updates the X weight vector \eqn{w = E^\top u}, normalized.
      \item Computes the X score \eqn{t = E w}, normalized.
      \item Updates the Y loading \eqn{q = F^\top t}, normalized.
      \item Updates the response score \eqn{u = F q}.
      \item Repeats until \eqn{t} converges below a tolerance threshold.
    }
  \item Computes scalar regression coefficient \eqn{b = t^\top u}.
  \item Deflates residual matrices \eqn{E} and \eqn{F} to remove current component contribution.
}

After component extraction, the final regression coefficient matrix \eqn{B_{original}} is computed and rescaled to the original
data units. Explained variance is also computed component-wise and cumulatively.
}
\examples{
\dontrun{
X <- matrix(rnorm(100 * 10), 100, 10)
Y <- matrix(rnorm(100 * 2), 100, 2)
model <- pls.regression(X, Y, n.components = 3, calc.method = "NIPALS")
model$coefficients
}

}
\references{
Wold, H., & Lyttkens, E. (1969). Nonlinear iterative partial least squares (NIPALS) estimation procedures. \emph{Bulletin of the International Statistical Institute}, 43, 29–51.
}
\keyword{internal}
