% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/kinship_fam.R
\name{kinship_fam}
\alias{kinship_fam}
\title{Calculate kinship matrix of a pedigree with structured founders}
\usage{
kinship_fam(kinship, fam, missing_vals = c("", 0))
}
\arguments{
\item{kinship}{The kinship matrix of the founders.
This matrix must have column and row names that identify each founder (matching codes in \code{fam$id}).
Individuals may be in a different order than \code{fam$id}.
Extra individuals in \code{kinship} but absent in \code{fam$id} will be silently ignored.
A traditional pedigree calculation would use \code{kinship = diag(n)/2} (plus appropriate column/row names), where \code{n} is the number of founders, to model unrelated and outbred founders.
However, if \code{kinship} measures the population kinship estimates between founders, the output is also a population kinship matrix (which combines the structural/ancestral and local/pedigree relatedness values into one).}

\item{fam}{The pedigree data.frame, in plink FAM format.
Only columns \code{id}, \code{pat}, and \code{mat} are required.
\code{id} must be unique and non-missing.
Founders must be present, and their \code{pat} and \code{mat} values must be missing (see below).
Non-founders must have both their parents be non-missing.
Parents must appear earlier than their children in the table.}

\item{missing_vals}{The list of ID values treated as missing.
\code{NA} is always treated as missing.
By default, the empty string ('') and zero (0) are also treated as missing (remove values from here if this is a problem).}
}
\value{
The kinship matrix of the entire \code{fam} table, taking the relatedness of the founders into account.
The rows and columns of this kinship matrix correspond to \code{fam$id} in that order.
}
\description{
Calculates a full kinship matrix (between all individuals in the provided pedigree FAM table) taking into account the relatedness of the founders as provided.
Output agrees with \code{\link[kinship2:kinship]{kinship2::kinship()}} but only when founders are unrelated/outbred (in other words, that function does not allow relatedness between founders).
}
\examples{
# The smallest pedigree, two parents and a child.
# A minimal fam table with the three required columns.
# Note "mother" and "father" have missing parent IDs, while "child" does not
library(tibble)
fam <- tibble(
  id = c('father', 'mother', 'child'),
  pat = c(NA, NA, 'father'),
  mat = c(NA, NA, 'mother')
)

# Kinship of the parents, here two unrelated/outbred individuals:
kinship <- diag(2)/2
# Name the parents with same codes as in `fam`
# (order can be different)
colnames( kinship ) <- c('mother', 'father')
rownames( kinship ) <- c('mother', 'father')
# For a clearer example, make the father slightly inbred
# (a self-kinship value that exceeds 1/2):
kinship[2,2] <- 0.6

# Calculate the full kinship matrix
kinship_all <- kinship_fam( kinship, fam )

# This is a 3x3 matrix with row/col names matching fam$id.
# The parent submatrix equals the input (reordered),
# but now there's relatedness to the child too
kinship_all

}
\seealso{
Plink FAM format reference:
\url{https://www.cog-genomics.org/plink/1.9/formats#fam}
}
