% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calc_burn_probability.R
\name{calc_abp}
\alias{calc_abp}
\title{Calculate annual burned probability (BP) from perimeters}
\usage{
calc_abp(template_raster, candidate_surfaces, reference_surface)
}
\arguments{
\item{template_raster}{A `terra::SpatRaster` object that defines the
extent, resolution, and CRS for the output burned probability raster.}

\item{candidate_surfaces}{An `sf` object (simple features) representing
the simulated fire perimeters. It is assumed to have a column named `size`
containing the area of each perimeter.}

\item{reference_surface}{Numeric value. A reference total surface area used
for normalization. This typically represents the target total burned area
(e.g., historical average annual burned area).}
}
\value{
A `terra::SpatRaster` object representing the burned probability.
  Each cell value indicates the proportion of times that pixel was burned,
  normalized by the ratio of total simulated area to the reference surface.
}
\description{
This function calculates the burned probability raster from a set of
simulated fire perimeters. It first converts the perimeters into a `terra`
vector object, ensures their validity, and then rasterizes them onto a
template raster. The resulting raster indicates the number of times each
pixel has been burned, which is then normalized by a scaling factor derived
from the total simulated area and a reference surface.
}
\examples{
# Create a dummy template raster
library(terra)
template_raster_ex <- rast(nrows=10, ncols=10, xmin=0, xmax=100, ymin=0, ymax=100,
                           crs="EPSG:25830")

# Create dummy candidate_surfaces (sf object with polygons)
# Ensure 'size' column exists
library(sf)
poly1 <- st_polygon(list(cbind(c(10,10,30,30,10), c(10,30,30,10,10))))
poly2 <- st_polygon(list(cbind(c(50,50,70,70,50), c(50,70,70,50,50))))
poly3 <- st_polygon(list(cbind(c(20,20,40,40,20), c(20,40,40,20,20))))

candidate_surfaces_ex <- st_sf(
  id = 1:3,
  size = c(400, 400, 400), # Dummy sizes for example
  geometry = st_sfc(poly1, poly2, poly3),
  crs = 25830
)

# Example reference surface
reference_surface_ex <- 1000

# Calculate burned probability
bp_raster <- calc_abp(template_raster = template_raster_ex,
                     candidate_surfaces = candidate_surfaces_ex,
                     reference_surface = reference_surface_ex)

plot(bp_raster, main = "Burned Probability Raster")
}
