#' Computes crude/raw death rates/probabilities using double rolling windows
#'
#' @description This function estimates (crude/raw) death rates/probabilities using
#'              rolling windows in two dimensions (usually, age and time) from a
#'              given experience of mortality, summarized in a dataset of (either initial
#'              or times of) exposed to risk and a dataset of recorded deaths.
#'
#' @author Jose M. Pavia \email{pavia@@uv.es}
#' @author Josep Lledo \email{josep.lledo@@uv.es}
#'
#' @param exposed A matrix (or data frame) of order nxT (where n is the number of ages and T the number of years), with ages by rows and years by columns (with no column for ages). For each age-year combination, the matrix contains either the initial number of individuals exposed to risk in the portfolio (population) or the time of exposition to risk of the exposed in the portfolio (population). By default, the matrix contains time exposures. When `exposed` contains initial number of individuals exposed to risk, time exposures are estimated assuming uniformity and a closed demographic system. It is assumed that the order of rows and columns in `deaths` and `exposed` is consistent  (i.e., identical), and that both ages and years are sequential.
#' @param deaths A matrix (or data frame) of order nxT, with ages by rows and years by columns (with no column for ages), containing the number of deaths recorded in the portfolio (population), where n is the number of ages and T the number of years. It is assumed that the order of rows and columns in `deaths` and `exposed` is consistent (i.e., are the same), and that both ages and years are sequential.
#' @param age.window An non-negative integer indicating the number of preceding and succeeding ages to be included in the age rolling window. Default, 2.
#' @param time.window An non-negative integer indicating the number of preceding and succeeding years (time periods) to be included in the year rolling window. Default, 2.
#' @param age.wb An non-negative integer indicating the number of preceding ages to be included in the age rolling window. Default, `age.window`.  If `age.wb` and `age.window` differ, the definition of `age.wb` takes precedence, allowing non-centered age windows.
#' @param age.wf An non-negative integer indicating the number of succeeding ages to be included in the age rolling window. Default, `age.window`. If `age.wf` and `age.window` differ, the definition of `age.wf` takes precedence, allowing non-centered age windows.
#' @param time.wb An non-negative integer indicating the number of preceding years to be included in the year rolling window. Default, `time.window`.  If `time.wb` and `time.window` differ, the definition of `time.wb` takes precedence, allowing non-centered time windows.
#' @param time.wf An non-negative integer indicating the number of succeeding years to be included in the year rolling window. Default, `time.window`. If `time.wf` and `time.window` differ, the definition of `time.wf` takes precedence, allowing non-centered time windows.
#' @param weights A matrix of order (age.wb + age.wf + 1) x (time.wb + time.wf + 1) of real positive numbers indicating the multiplicative factors to be applied to the different components of the double rolling window. Default, `1`; meaning all the components of the rolling window are aggregated as available in `exposed` and `deaths`.
#' @param initial A TRUE/FALSE argument indicating whether the numbers in `exposed` represent the initial number of individuals exposed to risk in the portfolio (population), which corresponds to `initial = TRUE`, or the time exposed to risk, with `initial = FALSE` as default.
#' @param partial A TRUE/FALSE argument indicating whether estimates obtained using incomplete rolling windows should be also computed. Default, `FALSE`; meaning crude death rates are estimated only for combinations of (year, age) where the corresponding double rolling window is complete.
#'
#' @return
#' A list with six matrices: `mx`, `qx`, `Lx`, `dx`, `Lx.total`, `dx.total`. When `partial = FALSE` these matrices are of order (n - age.wb - age.wf) x (T - time.wb - time.wf); otherwise, they are of order nxT. The names of the rows and of the columns of these matrices are inherited from the corresponding names in the `deaths` object, ensuring that if the row names indicate ages and the column names indicate years, this contextual information is preserved.
#'  \item{mx}{ The `mx` matrix holds the crude death rates estimated after determining the number of individuals (time) exposed to risk and deaths to be used for each estimate using the specified rolling windows with the defined `weights`. }
#'  \item{qx}{ The `qx` matrix contains the raw death probabilities, derived from `mx`, based on the assumption that, on average, each deceased individual lives for half a year in the year of their death. }
#'  \item{Lx}{ The `Lx` matrix contains the actual time exposures used to compute the estimates. }
#'  \item{dx}{ The `dx` matrix contains the number of deaths used to compute the estimates.. }
#'  \item{Lx.total}{ The `Lx.total` matrix contains the total numbers for time exposures which corresponds to all factor-weights being equal. If `weights` is a constant matrix `Lx` and `Lx.total` coincide. }
#'  \item{dx.total}{ The `dx.total` matrix contains the total numbers deaths which corresponds to all factor-weights being equal. If `weights` is a constant matrix `dx` and `dx.total` coincide. }
#'
#' @export
#'
#' @note The function could be used to apply double rolling windows to datasets from other fields other than demography and actuarial science.
#'
#' @examples
#'
#' exposed <- structure(list(Year2017 = c(6078.14, 5841.78, 5575.70, 5726.18, 5458.21, 5197.56,
#'                                     5018.12, 4791.56, 4245.15, 4321.65, 4179.3),
#'                           Year2018 = c(5978.73, 5473.78, 5572.23, 5495.19, 5148.47, 4845.14,
#'                                     4739.54, 4222.01, 4476.99, 4306.45, 4108.58),
#'                           Year2019 = c(5593.23, 5551.41, 5260.44, 5079.56, 4873.37, 4857.78,
#'                                     4536.12, 4453.85, 4310.89, 4015.02, 3974.25)),
#'                            class = "data.frame", row.names = 68:78)
#'
#' deaths <- structure(list(Year2017 = c(144, 102, 113, 122, 156, 110, 126, 132, 120, 172, 110),
#'                           Year2018 = c(111, 122, 109, 116, 162, 154, 115, 146, 100, 169, 146),
#'                            Year2019 = c(100, 123, 113, 151, 122, 110, 137, 175, 137, 110, 155)),
#'                          class = "data.frame", row.names = 68:78)
#'
#' example <- dw_crude_mx(exposed = exposed, deaths = deaths,
#'                        age.window = 2, time.window = 1, initial = FALSE)
#'

dw_crude_mx <- function(exposed,
                        deaths,
                        age.window = 2,
                        time.window = 2,
                        age.wb = age.window,
                        age.wf = age.window,
                        time.wb = time.window,
                        time.wf = time.window,
                        weights = 1,
                        initial = FALSE,
                        partial = FALSE){

  exposed <- as.matrix(exposed)
  deaths <- as.matrix(deaths)
  if (length(weights) == 1 & weights == 1)
    weights <- matrix(1L, nrow = age.wb + age.wf + 1L, ncol = time.wb + time.wf + 1L)

  # Test inputs
  args <- as.list(environment())
  test_dw(args)

  # Initial
  if (initial){
    exposed1 <- exposed/2
    exposed2 <- rbind(rep(0, ncol(exposed)), exposed1)[-(nrow(exposed) + 1L), ]
    exposed <- exposed1 + exposed2 - 0.5*deaths
  }

  # Expanded matrices
  nr <- nrow(deaths)
  nc <- ncol(deaths)
  exposed.a <- cbind(matrix(0, nrow = nr, ncol = time.wb), exposed,
                     matrix(0, nrow = nr, ncol = time.wf))
  deaths.a <- cbind(matrix(0, nrow = nr, ncol = time.wb), deaths,
                    matrix(0, nrow = nr, ncol = time.wf))
  exposed.a <- rbind(matrix(0, nrow = age.wb , ncol = nc + time.wb + time.wf), exposed.a,
                     matrix(0, nrow = age.wf, ncol = nc + time.wb + time.wf))
  deaths.a <- rbind(matrix(0, nrow = age.wb, ncol = nc + time.wb + time.wf), deaths.a,
                    matrix(0, nrow = age.wf, ncol = nc + time.wb + time.wf))

  # Windows aggregation of exposed and deaths
  Lx <- dx <- Lx.total <- dx.total <- deaths
  weights.total <- weights
  weights.total[] <- 1

  for (ii in 1L:nr){
    for (jj in 1L:nc){
      temp.e <- exposed.a[ii:(ii + age.wb + age.wf), jj:(jj + time.wb + time.wf)]
      temp.d <- deaths.a[ii:(ii + age.wb + age.wf), jj:(jj + time.wb + time.wf)]
      Lx[ii, jj] <- sum(temp.e * weights)
      Lx.total[ii, jj] <- sum(temp.e * weights.total)
      dx[ii, jj] <- sum(temp.d * weights)
      dx.total[ii, jj] <- sum(temp.d * weights.total)
    }
  }
  mx <- dx/Lx
  qx <- mx/(1 + 0.5*mx)
  if(!partial){
    mx <- mx[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
    qx <- qx[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
    Lx <- Lx[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
    dx <- dx[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
    Lx.total <- Lx.total[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
    dx.total <- dx.total[(1L + age.wb):(nr - age.wf), (1L + time.wb):(nc - time.wf)]
  }

  output <- list("mx" = mx, "qx" = qx, "Lx" = Lx, "dx" = dx,
                 "Lx.total" = Lx.total, "dx.total" = dx.total)
  class(output) <- c("rwlifetable", "dw_crude_mx", "list")
  return(output)
}




test_dw <- function(args){
  if (!all(dim(args$exposed) == dim(args$deaths)))
    stop("The number of rows and/or columns in 'exposed' and 'deaths' differ.")

  if (min(args$exposed) < 0 | min(args$deaths) < 0)
    stop("Negative numbers are not allowed in 'exposed' or 'deaths'.")

  if ((args$age.window - floor(args$age.window)) > 0 | args$age.window < 0)
    stop("The argument 'age.window' must be a non-negative integer.")

  if ((args$time.window - floor(args$time.window)) > 0 | args$time.window < 0)
    stop("The argument 'time.window' must be a non-negative integer.")

  if ((args$age.wb - floor(args$age.wb)) > 0 | args$age.wb < 0)
    stop("The argument 'age.wb' must be a non-negative integer.")

  if ((args$age.wf - floor(args$age.wf)) > 0 | args$age.wf < 0)
    stop("The argument 'age.wf' must be a non-negative integer.")

  if ((args$time.wb - floor(args$time.wb)) > 0 | args$time.wb < 0)
    stop("The argument 'time.wb' must be a non-negative integer.")

  if ((args$time.wf - floor(args$time.wf)) > 0 | args$time.wf < 0)
    stop("The argument 'time.wf' must be a non-negative integer.")

  if (!all(dim(args$weights) == c(args$age.wb + args$age.wf + 1L, args$time.wb + args$time.wf + 1L)))
    stop("The argument 'weights' must be 1 or a matrix of order (age.wb + age.wf + 1) x (time.wb + time.wf + 1).")

  if (min(args$weights) < 0)
    stop("The values in 'weights' cannot be negative.")

  if (!args$partial & (args$age.wb + args$age.wf + 1) > nrow(args$deaths))
    stop("The age rolling window is too large considering the number of ages in the datasets.")

  if (!args$partial & (args$time.wb + args$time.wf + 1) > ncol(args$deaths))
    stop("The time rolling window is too large considering the number of years in the datasets.")
}
