#' Check SNV reversions for exonic SNV
#'
#' @param roi.aln.df A data frame of the detection window alignment 
#' @param minus.strand A logical value indicating whether the pathogenic mutation on the minus strand
#' @param pathog.mut.alt The alternative allele of the pathogenic SNV
#' @param tx.id The Ensembl Transcript ID for the pathogenic SNV
#' @param ensdb An EnsDb object of Ensembl-based annotation database
#' 
#' @return A vector of genomic positions
#' 
#' @importFrom Biostrings DNAString
#' @importFrom Biostrings reverseComplement
#' @importFrom Biostrings translate
#'
#' @examples
#' \dontrun{
#'     checkExonicSNV()
#' }
#'
#' @noRd
checkExonicSNV <- function(roi.aln.df, minus.strand, pathog.mut.alt, tx.id, ensdb){
    
    cryptic.snv <- NULL
    is.stop.lost.rev <- FALSE
    is.non.sense.rev <- FALSE
    is.mis.sense.rev <- FALSE
    aa.codes <- unlist(strsplit("ACDEFGHIKLMNPQRSTVWY",""))
    
    snv.base <- roi.aln.df$ref_pos[ roi.aln.df$pathog_mut==1 ]
    snv.base.cds <- genomeToCds(snv.base, tx.id, ensdb)
    
    if( snv.base.cds%%3 == 1 ){
        snv.window <- c( 
            snv.base, 
            cdsToGenome(snv.base.cds+1, tx.id, ensdb),
            cdsToGenome(snv.base.cds+2, tx.id, ensdb)
        )
    }else if( snv.base.cds%%3 == 2 ){
        snv.window <- c( 
            cdsToGenome(snv.base.cds-1, tx.id, ensdb), 
            snv.base, 
            cdsToGenome(snv.base.cds+1, tx.id, ensdb) 
        )
    }else if( snv.base.cds%%3 == 0 ){
        snv.window <- c( 
            cdsToGenome(snv.base.cds-2, tx.id, ensdb), 
            cdsToGenome(snv.base.cds-1, tx.id, ensdb), 
            snv.base 
        )
    }
    
    snv.window <- sort(snv.window)
    check.df <- roi.aln.df[ roi.aln.df$ref_pos %in% snv.window, ]
    
    if( nrow(check.df)==3 & sum(check.df$cigar=="M")==3 & sum(is.na(check.df$ins[setdiff(1:nrow(check.df), nrow(check.df))]))==2 ){
        
        ref.codon <- paste( check.df$ref_seq, collapse="" )
        mut.codon <- paste( ifelse(check.df$pathog_mut==1, pathog.mut.alt, check.df$ref_seq), collapse="" )
        rev.codon <- paste( check.df$reads_seq, collapse="" )
            
        if(!minus.strand){
            ref.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::DNAString(ref.codon), if.fuzzy.codon="X")) )
            mut.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::DNAString(mut.codon), if.fuzzy.codon="X")) )
            rev.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::DNAString(rev.codon), if.fuzzy.codon="X")) )
        }else{
            ref.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::reverseComplement(Biostrings::DNAString(ref.codon)), if.fuzzy.codon="X")) )
            mut.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::reverseComplement(Biostrings::DNAString(mut.codon)), if.fuzzy.codon="X")) )
            rev.aa <- suppressWarnings( as.character(Biostrings::translate(Biostrings::reverseComplement(Biostrings::DNAString(rev.codon)), if.fuzzy.codon="X")) )
        }
        
        is.stop.lost.rev <- (ref.aa=="*") & (mut.aa%in%aa.codes) & (rev.aa=="*")
        is.non.sense.rev <- (ref.aa%in%aa.codes) & (mut.aa=="*") & (rev.aa%in%aa.codes)
        is.mis.sense.rev <- (ref.aa%in%aa.codes) & (mut.aa%in%aa.codes) & (rev.aa%in%aa.codes) & (ref.aa!=mut.aa) & (mut.aa!=rev.aa)
        
        if( is.stop.lost.rev | is.non.sense.rev | is.mis.sense.rev ){
            cryptic.snv <- check.df$ref_pos[ !is.na( check.df$snv ) ]
            if( check.df$reads_seq[check.df$pathog_mut==1]==pathog.mut.alt ){
                cryptic.snv <- setdiff( cryptic.snv, snv.base )
                if(length(cryptic.snv)==0){
                    cryptic.snv <- NULL
                }
            }
        }
        
    }
    
    return(cryptic.snv)
    
}



