#' Realign a soft-clipped read based on the original alignment
#'
#' @param reads The sequence of a soft-clipped read
#' @param reference.genome A BSgenome or DNAStringSet object of reference sequences
#' @param chr.name The chromosome name or sequence name of the reference sequence
#' @param ref.start The start position of reference sequence in the original alignment
#' @param ref.end The end position of reference sequence in the original alignment
#' @param match The scoring for a nucleotide match in the alignment
#' @param mismatch The scoring for a nucleotide mismatch in the alignment
#' @param gapOpening The cost for opening a gap in the alignment
#' @param gapExtension The incremental cost incurred along the length of the gap in the alignment
#' @param ref.flank.left The length of flanking region to be added to the left end of original alignment region
#' @param ref.flank.right The length of flanking region to be added to the right end of original alignment region
#' @param indel.shifting A character variable specifying the direction (left or right) for shifting the start position of indels till it is no longer possible to do so
#' 
#' @return A list of the soft-clipped read realignment including cigar, start position of reference sequence, end position of reference sequence
#' 
#' @importFrom BSgenome getSeq
#' @importFrom Biostrings matchPattern
#' @importFrom GenomicRanges GRanges
#' @importFrom IRanges IRanges
#' @importFrom IRanges ranges
#' @importFrom BiocGenerics start
#'
#' @examples
#' \dontrun{
#'     alignSoftClippedRead()
#' }
#'
#' @noRd
alignSoftClippedRead <- function(
        reads, 
        reference.genome, 
        chr.name, 
        ref.start, 
        ref.end, 
        match=1, 
        mismatch=4, 
        gapOpening=6, 
        gapExtension=0, 
        ref.flank.left=1000,
        ref.flank.right=1000,
        indel.shifting
    ){
    chr.length <- length( unlist( BSgenome::getSeq(reference.genome, chr.name) ) )
	ref.start <- max( ref.start-(ref.flank.left+ref.flank.right), 1 ) 
	ref.end <- min( ref.end+(ref.flank.left+ref.flank.right), chr.length ) 
	ref <- as.character( BSgenome::getSeq(reference.genome, GenomicRanges::GRanges(chr.name, IRanges::IRanges(ref.start,ref.end)) ) )
	aln <- gapShiftingPairwiseAlignment(ref, reads, indel.shifting, "local-global", match, mismatch, gapOpening, gapExtension)
	cigar <- encodeCIGAR(aln[1], aln[2])
	aln.pos.relative <- BiocGenerics::start( IRanges::ranges(Biostrings::matchPattern(gsub("-","",aln[1]), ref)) )[1]
	aln.pos <- ref.start + aln.pos.relative - 1
	ref.start <- aln.pos
	ref.end <- aln.pos + nchar(gsub("-","",aln[1])) - 1 
	res <- list(
		cigar = cigar,
		ref.start = ref.start,
		ref.end = ref.end
	)
	return(res)
}
