% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/rdpmedian.r
\name{rdpmedian}
\alias{rdpmedian}
\alias{rdpmedian2}
\title{Rank difference pseudomedian}
\usage{
rdpmedian(
  data,
  formula,
  conf_level = 0.95,
  conf_method = "percentile",
  n_resamples = 1000L,
  agg_fun = "error"
)

rdpmedian2(
  x,
  y,
  conf_level = 0.95,
  conf_method = "percentile",
  n_resamples = 1000L
)
}
\arguments{
\item{data}{(data.frame)\cr
The data frame of interest.}

\item{formula}{(formula)\cr
A formula of form:
\describe{
\item{y ~ group | block}{
Use when \code{data} is in tall format.
\code{y} is the numeric outcome, \code{group} is the binary grouping variable, and \code{block} is the subject/item-level variable indicating pairs of observations.
\code{group} will be converted to a factor and the first level will be the reference value.
For example, when \code{levels(data$group) <- c("pre", "post")}, the focal level is 'post', so differences are \code{post - pre}.
Pairs with missing values are silently dropped.
See \code{agg_fun} for handling duplicate cases of grouping/blocking combinations.
}
\item{y ~ x}{
Use when \code{data} is in wide format.
\code{y} and \code{x} must be numeric vectors.
Differences of ranks correspond with \code{data$y - data$x}.
Pairs with missing values are silently dropped.
}
}}

\item{conf_level}{(Scalar numeric: \code{0.95}; \verb{[0, 1)})\cr
The confidence level.
If \code{conf_level = 0}, no confidence interval is calculated.}

\item{conf_method}{(Scalar character: \code{c("percentile", "bca")})\cr
The type of bootstrap confidence interval.}

\item{n_resamples}{(Scalar integer: \verb{[10L, Inf)}\cr
The number of bootstrap resamples.
If \code{conf_level = 0}, no resampling is performed.}

\item{agg_fun}{(Scalar character or function: \code{"error"})\cr
Used for aggregating duplicate cases of grouping/blocking combinations when data is in tall format and \code{formula} has structure \code{y ~ group | block}.
\code{"error"} (default) will return an error if duplicate grouping/blocking combinations are encountered.
Select one of \code{"first"}, \code{"last"}, \code{"sum"}, \code{"mean"}, \code{"median"}, \code{"min"}, or \code{"max"} for built in aggregation handling (each applies \code{na.rm = TRUE}).
Or define your own function.
For example, \code{myfun <- function(x) {as.numeric(quantile(x, 0.75, na.rm = TRUE))}}.}

\item{x}{(numeric)\cr
Numeric vector of data.
Differences of ranks correspond with \code{x - y}.
Pairs with missing values are silently dropped.}

\item{y}{(numeric)\cr
Numeric vector of data.
Differences of ranks correspond with \code{x - y}.
Pairs with missing values are silently dropped.}
}
\value{
A list with the following elements:
\tabular{llll}{
Slot \tab Subslot \tab Name \tab Description \cr
1 \tab \tab \code{pseudomedian} \tab Measure of centrality. \cr
2 \tab \tab \code{lower}        \tab Lower bound of confidence interval for the pseudomedian. \cr
3 \tab \tab \code{upper}        \tab Upper bound of confidence interval for the pseudomedian. \cr
4 \tab \tab \code{method}       \tab Estimate method. \cr
5 \tab   \tab \code{info} \tab Additional information. \cr
5 \tab 1 \tab \code{n_sample} \tab Number of observations in the original data. \cr
5 \tab 2 \tab \code{n_analytic} \tab Number of observations after removing missing values from the original data. \cr
5 \tab 3 \tab \code{data_type} \tab Data type. \cr
5 \tab 4 \tab \code{focal_name} \tab Name of the focal variable (differences are focal - reference). \cr
5 \tab 5 \tab \code{reference_name} \tab Name of the reference variable (differences are focal - reference). \cr
6 \tab   \tab \code{call} \tab A named list of the function's arguments (use \code{as.call()} to convert to a call).
}
}
\description{
Computes the Hodges-Lehmann pseudomedian and bootstrap confidence interval for the differences of ranks.
}
\details{
This function generates a confidence interval for the pseudomedian based on the observed differences of ranks, not based on an inversion of the rank difference test \code{\link[=rdt]{rdt()}}.

The Hodges-Lehmann estimator is the median of all pairwise averages of the sample values.
\deqn{\mathrm{HL} = \mathrm{median} \left\{ \frac{x_i + x_j}{2} \right\}_{i \le j}}
This pseudomedian is a robust, distribution-free estimate of central tendency for a single sample, or a location-shift estimator for paired data.
It's resistant to outliers and compatible with rank-based inference.

The percentile and BCa bootstrap confidence interval methods are described in chapter 5.3 of \insertCite{davison1997;textual}{rankdifferencetest}.

This function is mainly a wrapper for the function \code{\link[Hmisc:pMedian]{Hmisc::pMedian()}}.
}
\examples{
#----------------------------------------------------------------------------
# rdpmedian() example
#----------------------------------------------------------------------------
library(rankdifferencetest)

# Use example data from Kornbrot (1990)
data <- kornbrot_table1

# Create tall-format data for demonstration purposes
data_tall <- reshape(
  data = kornbrot_table1,
  direction = "long",
  varying = c("placebo", "drug"),
  v.names = c("time"),
  idvar = "subject",
  times = c("placebo", "drug"),
  timevar = "treatment",
  new.row.names = seq_len(prod(length(c("placebo", "drug")), nrow(kornbrot_table1)))
)

# Subject and treatment should be factors. The ordering of the treatment factor
# will determine the difference (placebo - drug).
data_tall$subject <- factor(data_tall$subject)
data_tall$treatment <- factor(data_tall$treatment, levels = c("drug", "placebo"))

# Rate transformation inverts the rank ordering.
data$placebo_rate <- 60 / data$placebo
data$drug_rate <- 60 / data$drug
data_tall$rate <- 60 / data_tall$time

# Estimates
rdpmedian(
  data = data,
  formula = placebo ~ drug
)

rdpmedian(
  data = data_tall,
  formula = time ~ treatment | subject
)

rdpmedian2(
  x = data$placebo_rate,
  y = data$drug_rate
)

rdpmedian(
  data = data_tall,
  formula = rate ~ treatment | subject
)

}
\references{
\insertRef{davison1997}{rankdifferencetest}

\insertRef{Hmisc}{rankdifferencetest}
}
\seealso{
\code{\link[=pmedian]{pmedian()}}
}
