#' QuantBayES Universal Bayesian Evidence Sufficiency Scoring
#'
#' @description
#' Computes posterior theta, credible intervals and percentiles for each 
#' variant given a binary evidence matrix.
#'
#' @param x matrix of 0 and 1, rows are variants and columns are evidence rules.
#' @param a prior alpha parameter.
#' @param b prior beta parameter.
#' @param ci_level credible interval width.
#'
#' @return A list with:
#' \describe{
#'   \item{variants}{Data frame of per variant scores.}
#'   \item{global}{Summary list of global posterior distribution.}
#' }
#'
#' @importFrom stats median
#' @export
quant_es_core <- function(x, a = 1, b = 1, ci_level = 0.95) {
  
  if (!is.matrix(x)) stop("Input x must be a matrix.")
  if (!is.numeric(x)) stop("Input matrix must contain numeric values.")
  
  na_count <- sum(is.na(x))
  if (na_count > 0) {
    message("Quant ES core: converting ", na_count, " NA values to 0.")
    x[is.na(x)] <- 0
  }
  if (!all(x %in% c(0, 1))) stop("Matrix must contain only 0 or 1.")
  
  n_variants <- nrow(x)
  m_rules <- ncol(x)
  
  k <- rowSums(x)
  alpha <- a + k
  beta <- b + m_rules - k
  
  theta_mean <- alpha / (alpha + beta)
  
  lower_q <- (1 - ci_level) / 2
  upper_q <- 1 - lower_q
  
  theta_lower <- stats::qbeta(lower_q, alpha, beta)
  theta_upper <- stats::qbeta(upper_q, alpha, beta)
  
  percentile <- 100 * rank(theta_mean, ties.method = "average") / n_variants
  
  variant_id <- rownames(x)
  if (is.null(variant_id)) variant_id <- seq_len(n_variants)
  
  variants <- tibble::tibble(
    variant_id,
    k,
    m = m_rules,
    theta_mean,
    theta_lower,
    theta_upper,
    percentile
  )
  
  global <- list(
    n_variants = n_variants,
    m_rules = m_rules,
    mean_theta = mean(theta_mean),
    median_theta = median(theta_mean),
    lower_theta = stats::quantile(theta_mean, probs = lower_q),
    upper_theta = stats::quantile(theta_mean, probs = upper_q),
    ci_level = ci_level
  )
  
  list(variants = variants, global = global)
}
#' @examples
#' data(core_test_data)
#' x <- as.matrix(core_test_data[, -1])
#' rownames(x) <- core_test_data[[1]]
#'
#' res <- quant_es_core(x)
#' res$global
#' head(res$variants)
