#' Global Optimization of Approximation Error
#'
#' This function computes the global minimum and maximum of the absolute error
#' between the target function `f` and its piecewise linear (PWL) approximation
#' across each linear segment.
#'
#' @param PWL A matrix with 4 columns: slope, intercept, lower bound, upper bound.
#' @param f The original function to approximate.
#' @param tol Tolerance for optimizer convergence.
#'
#' @return A matrix with 4 columns:
#'   x_min (location of min error), min_error,
#'   x_max (location of max error), max_error.
#' @keywords internal
find_global_optima <- function(PWL, f, tol = 1e-6) {
  if (!requireNamespace("nloptr", quietly = TRUE)) {
    stop("Package 'nloptr' is required but not installed.")
  }

  num_segments <- nrow(PWL)
  results <- matrix(0, nrow = num_segments, ncol = 4)

  for (i in 1:num_segments) {
    slope <- PWL[i, 1]
    intercept <- PWL[i, 2]
    lb <- PWL[i, 3]
    ub <- PWL[i, 4]

    # Objective: Absolute error between true function and PWL line
    eval_error <- function(x) {
      abs(f(x[1]) - (slope * x[1] + intercept))
    }

    # Minimize error
    res_min <- tryCatch({
      nloptr::nloptr(
        x0 = c((lb + ub)/2),
        eval_f = eval_error,
        lb = c(lb),
        ub = c(ub),
        opts = list(
          algorithm = "NLOPT_GN_DIRECT_L_RAND_NOSCAL",
          xtol_rel = tol,
          maxeval = 1000
        )
      )
    }, error = function(e) NULL)

    # Maximize error (by minimizing negative error)
    eval_neg_error <- function(x) {
      -eval_error(x)
    }

    res_max <- tryCatch({
      nloptr::nloptr(
        x0 = c((lb + ub)/2),
        eval_f = eval_neg_error,
        lb = c(lb),
        ub = c(ub),
        opts = list(
          algorithm = "NLOPT_GN_DIRECT_L_RAND_NOSCAL",
          xtol_rel = tol,
          maxeval = 1000
        )
      )
    }, error = function(e) NULL)

    # Store results
    if (!is.null(res_min)) {
      results[i, 1] <- res_min$solution
      results[i, 2] <- res_min$objective
    } else {
      results[i, 1:2] <- NA
      warning(sprintf("Minimization failed on interval [%f, %f]", lb, ub))
    }

    if (!is.null(res_max)) {
      results[i, 3] <- res_max$solution
      results[i, 4] <- -res_max$objective
    } else {
      results[i, 3:4] <- NA
      warning(sprintf("Maximization failed on interval [%f, %f]", lb, ub))
    }

    # Skip degenerate intervals
    if ((ub < lb || abs(ub - lb) < 1e-8) && (is.null(res_min) || is.null(res_max))) {
      results[i, ] <- rep(NA, 4)
      message(sprintf("Skipped degenerate interval: [%f, %f]", lb, ub))
    }
  }

  return(results)
}
