#' @title Search VecTraits by keyword
#' @description Retrieve the IDs for any VecTraits datasets matching the given keywords.
#'
#' @author Francis Windram
#'
#' @param keywords either a string of search terms separated by spaces, or a vector of keywords.
#' @param basereq an [httr2 request][httr2::request()] object, as generated by [vb_basereq()]. If `NA`, uses the default request.
#'
#' @return An `ohvbd.ids` vector of VecTraits dataset IDs.
#'
#' @examplesIf interactive()
#' search_vt("Aedes aegypti")
#'
#' search_vt(c("Aedes", "aegypti"))
#'
#' @note
#' [search_hub()] is now preferred for keyword searches:
#'
#' ```
#' # old style
#' search_vt(c("Ixodes", "ricinus")
#'
#' # new style
#' search_hub("Ixodes ricinus", db = "vt")
#' ```
#'
#' `search_vt()` may be deprecated in the future.
#'
#' @section Warning:
#' The ids returned from the server (and thus this function) do not necessarily precisely match the keywords that were requested.
#'
#' For example `search_vt("United Kingdom")` does not return only items found in the United Kingdom. Instead it returns items where some part of the string "United Kingdom" appears in one of the indexed columns.
#'
#' The indexed columns of VecTraits are:
#' - `DatasetID`
#' - `OriginalTraitName`
#' - `Variables`
#' - `Interactor1Order`
#' - `Interactor1Family`
#' - `Interactor1Genus`
#' - `Interactor1Species`
#' - `Interactor1Stage`
#' - `Interactor1Sex`
#' - `Interactor2Genus`
#' - `Interactor2Species`
#' - `Citation`
#' - `DOI`
#' - `CuratedByDOI`
#' - `SubmittedBy`
#'
#' @concept vectraits
#'
#' @export
#'

search_vt <- function(keywords, basereq = vb_basereq()) {
  cli::cli_inform(c(
    "i" = "{.fn ohvbd::search_hub} is now preferred for keyword searches:",
    "",
    '{.code search_hub("{paste(keywords, collapse = " ")}", db = "vt")}',
    "",
    "!" = "{.fn search_vt} may be deprecated in the future."),
    .frequency = "regularly",
    .frequency_id = "search_vt_deprecation_info")

  req <- basereq |>
    req_url_path_append("vectraits-explorer") |>
    req_url_query("format" = "json") |>
    req_url_query("keywords" = keywords, .multi = space_collapse)

  if (getOption("ohvbd_dryrun", default = FALSE)) {
    cli::cli_alert_warning("Debug option {.val ohvbd_dryrun} is TRUE.")
    cli::cli_alert_info("Returning request object...")
    return(req)
  }

  resplist <- tryCatch(
    {
      resp <- req |>
        req_perform()
      list("resp" = resp, "err_code" = 0, "err_obj" = NULL)
    },
    error = function(e) {
      # Get the last response instead
      list("resp" = last_response(), "err_code" = 1, "err_obj" = e)
    }
  )

  if (resplist$err_code == 1) {
    cli::cli_abort("No records found for {.val {keywords}}")
  }

  body <- resplist$resp |> resp_body_json()
  outids <- as.numeric(body$ids)
  outids <- new_ohvbd.ids(v = outids, db = "vt")
  return(outids)
}
