# mirai x parallel -------------------------------------------------------------

#' Make Mirai Cluster
#'
#' `make_cluster` creates a cluster of type 'miraiCluster', which may be used as
#' a cluster object for any function in the \pkg{parallel} base package such as
#' [parallel::clusterApply()] or [parallel::parLapply()].
#'
#' For R version 4.5 or newer, [parallel::makeCluster()] specifying
#' `type = "MIRAI"` is equivalent to this function.
#'
#' @param n (integer) number of nodes. Launched locally unless `url` is
#'   supplied.
#' @param url (character) host URL for remote nodes to dial into, e.g.
#'   'tcp://10.75.37.40:5555'. Use 'tls+tcp://' for secure TLS.
#' @param remote (configuration) for launching remote nodes, generated by
#'   [ssh_config()], [cluster_config()], or [remote_config()].
#' @param ... (daemons arguments) passed to [daemons()].
#'
#' @return For **make_cluster**: An object of class 'miraiCluster' and
#'   'cluster'. Each 'miraiCluster' has an automatically assigned ID and `n`
#'   nodes of class 'miraiNode'. If `url` is supplied but not `remote`, the
#'   shell commands for deployment of nodes on remote resources are printed to
#'   the console.
#'
#'   For **stop_cluster**: invisible NULL.
#'
#' @section Remote Nodes:
#'
#' Specify `url` and `n` to set up a host connection for remote nodes to dial
#' into. `n` defaults to one if not specified.
#'
#' Also specify `remote` to launch the nodes using a configuration generated by
#' [remote_config()] or [ssh_config()]. In this case, the number of nodes is
#' inferred from the configuration provided and `n` is disregarded.
#'
#' If `remote` is not supplied, the shell commands for deploying nodes manually
#' on remote resources are automatically printed to the console.
#'
#' [launch_remote()] may be called at any time on a 'miraiCluster' to return the
#' shell commands for deployment of all nodes, or on a 'miraiNode' to return the
#' command for a single node.
#'
#' @section Errors:
#'
#' Errors are thrown by the \pkg{parallel} package mechanism if one or more
#' nodes failed (quit unexpectedly). The returned 'errorValue' is 19
#' (Connection reset). Other types of error, e.g. in evaluation, result in the
#' usual 'miraiError' being returned.
#'
#' @note The default behaviour of clusters created by this function is designed
#'   to map as closely as possible to clusters created by the \pkg{parallel}
#'   package. However, `...` arguments are passed to [daemons()] for additional
#'   customisation, and not all combinations may be supported by \pkg{parallel}
#'   functions.
#'
#' @examplesIf interactive()
#' cl <- make_cluster(2)
#' cl
#' cl[[1L]]
#'
#' Sys.sleep(0.5)
#' status(cl)
#'
#' stop_cluster(cl)
#'
#' @export
#'
make_cluster <- function(n, url = NULL, remote = NULL, ...) {
  id <- sprintf("`%d`", length(..))

  if (is.character(url)) {
    daemons(n, url = url, remote = remote, dispatcher = FALSE, ..., cleanup = FALSE, .compute = id)

    if (is.null(remote)) {
      if (missing(n)) {
        n <- 1L
      }
      is.numeric(n) || stop(._[["numeric_n"]])
      cat("Shell commands for deployment on nodes:\n\n", file = stdout())
      print(launch_remote(n, .compute = id))
    } else {
      args <- remote[["args"]]
      n <- if (is.list(args)) length(args) else 1L
    }
  } else {
    is.numeric(n) || stop(._[["numeric_n"]])
    n >= 1L || stop(._[["n_one"]])
    daemons(n, dispatcher = FALSE, ..., cleanup = FALSE, .compute = id)
  }

  cl <- lapply(seq_len(n), create_node, id = id)
  `attributes<-`(cl, list(class = c("miraiCluster", "cluster"), id = id))
}

#' Stop Mirai Cluster
#'
#' `stop_cluster` stops a cluster created by `make_cluster`.
#'
#' @param cl (miraiCluster) cluster to stop.
#'
#' @rdname make_cluster
#' @export
#'
stop_cluster <- function(cl) {
  daemons(0L, .compute = attr(cl, "id"))
  invisible()
}

#' @exportS3Method parallel::stopCluster
#'
stopCluster.miraiCluster <- stop_cluster

#' @exportS3Method parallel::sendData
#'
sendData.miraiNode <- function(node, data) {
  id <- attr(node, "id")
  envir <- ..[[id]]
  is.null(envir) && stop(._[["cluster_inactive"]])

  value <- data[["data"]]
  tagged <- !is.null(value[["tag"]])
  if (tagged) {
    cv_reset(envir[["cv"]])
  }

  m <- mirai(
    do.call(node, data, quote = TRUE),
    node = value[["fun"]],
    data = value[["args"]],
    .compute = id
  )
  if (tagged) {
    `[[<-`(m, "tag", value[["tag"]])
  }
  `[[<-`(node, "mirai", m)
}

#' @exportS3Method parallel::recvData
#'
recvData.miraiNode <- function(node) call_aio(.subset2(node, "mirai"))

#' @exportS3Method parallel::recvOneData
#'
recvOneData.miraiCluster <- function(cl) {
  wait(..[[attr(cl, "id")]][["cv"]])
  node <- which.min(lapply(cl, node_unresolved))
  m <- .subset2(.subset2(cl, node), "mirai")
  list(node = node, value = `class<-`(m, NULL))
}

#' @export
#'
print.miraiCluster <- function(x, ...) {
  id <- attr(x, "id")
  cat(
    sprintf("< miraiCluster | ID: %s nodes: %d active: %s >\n", id, length(x), !is.null(..[[id]])),
    file = stdout()
  )
  invisible(x)
}

#' @export
#'
`[.miraiCluster` <- function(x, ...) .subset(x, ...)

#' @export
#'
print.miraiNode <- function(x, ...) {
  cat(
    sprintf("< miraiNode | node: %d cluster ID: %s >\n", attr(x, "node"), attr(x, "id")),
    file = stdout()
  )
  invisible(x)
}

# internals --------------------------------------------------------------------

create_node <- function(node, id) {
  `attributes<-`(
    new.env(hash = FALSE, parent = emptyenv()),
    list(class = "miraiNode", node = node, id = id)
  )
}

node_unresolved <- function(node) {
  m <- .subset2(node, "mirai")
  unresolved(m) || !is.object(m)
}
