% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mhp.R
\name{hp_filter}
\alias{hp_filter}
\title{Standard Hodrick-Prescott Filter}
\usage{
hp_filter(x, lambda = 1600, as_dt = TRUE)
}
\arguments{
\item{x}{Numeric vector. The time series to decompose.}

\item{lambda}{Numeric. The smoothing parameter. Default 1600 (quarterly data).
Common values: 1600 (quarterly), 100 (annual), 14400 (monthly).}

\item{as_dt}{Logical. If TRUE (default), returns a data.table. If FALSE,
returns a list.}
}
\value{
If \code{as_dt = TRUE}: A \code{data.table} with columns:
\describe{
  \item{original}{The input series}
  \item{trend}{The estimated trend component}
  \item{cycle}{The cyclical component}
}
With attribute \code{lambda} (the input lambda value).

If \code{as_dt = FALSE}: A list containing \code{original}, \code{trend},
\code{cycle}, and \code{lambda}.
}
\description{
Decomposes a time series into trend and cyclical components using the
standard HP filter with a fixed smoothing parameter lambda.
}
\details{
The HP filter solves the minimization problem:

\deqn{\min_{\{g_t\}} \left\{ \sum_{t=1}^T (y_t - g_t)^2 + \lambda \sum_{t=2}^{T-1} [(g_{t+1} - g_t) - (g_t - g_{t-1})]^2 \right\}}

The solution is obtained by solving:

\deqn{(I + \lambda K'K)g = y}

where \eqn{K} is the second-difference matrix.
}
\examples{
# Example 1: Simple random walk with cycle
set.seed(123)
n <- 80
y <- cumsum(rnorm(n)) + sin((1:n) * pi / 10)
result <- hp_filter(y, lambda = 1600)
head(result)

# Example 2: GDP-like series
set.seed(456)
gdp <- cumsum(rnorm(100, mean = 0.5, sd = 0.3)) + 2 * cos(2 * pi * (1:100) / 40)
gdp_decomp <- hp_filter(gdp, lambda = 1600)

# Plot the decomposition
if (require(ggplot2)) {
  plot_data <- data.table::data.table(
    t = 1:length(gdp),
    Original = gdp,
    Trend = gdp_decomp$trend,
    Cycle = gdp_decomp$cycle
  )
  plot_data_long <- data.table::melt(plot_data, id.vars = "t")

  ggplot2::ggplot(plot_data_long, ggplot2::aes(x = t, y = value, color = variable)) +
    ggplot2::geom_line(linewidth = 0.8) +
    ggplot2::facet_wrap(~variable, ncol = 1, scales = "free_y") +
    ggplot2::labs(
      title = "HP Filter Decomposition (lambda = 1600)",
      x = "Time", y = "Value"
    ) +
    ggplot2::theme_minimal() +
    ggplot2::theme(legend.position = "none")
}

}
\references{
Hodrick, R.J., & Prescott, E.C. (1997). Postwar US business cycles: An
empirical investigation. \emph{Journal of Money, Credit and Banking}, 29(1), 1-16.

Ravn, M.O., & Uhlig, H. (2002). On adjusting the Hodrick-Prescott filter
for the frequency of observations. \emph{Review of Economics and Statistics}, 84(2), 371-376.
}
