/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#define H5A_FRIEND     
#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Apkg.h"      
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Opkg.h"      

static void  *H5O__ainfo_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                size_t p_size, const uint8_t *p);
static herr_t H5O__ainfo_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                const void *_mesg);
static void  *H5O__ainfo_copy(const void *_mesg, void *_dest);
static size_t H5O__ainfo_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__ainfo_free(void *_mesg);
static herr_t H5O__ainfo_delete(H5F_t *f, H5O_t *open_oh, void *_mesg);
static herr_t H5O__ainfo_pre_copy_file(H5F_t *file_src, const void *mesg_src, bool *deleted,
                                       const H5O_copy_t *cpy_info, void *udata);
static void  *H5O__ainfo_copy_file(H5F_t *file_src, void *mesg_src, H5F_t *file_dst, bool *recompute_size,
                                   unsigned *mesg_flags, H5O_copy_t *cpy_info, void *udata);
static herr_t H5O__ainfo_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                                        void *mesg_dst, unsigned *mesg_flags, H5O_copy_t *cpy_info);
static herr_t H5O__ainfo_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_AINFO[1] = {{
    H5O_AINFO_ID,              
    "ainfo",                   
    sizeof(H5O_ainfo_t),       
    0,                         
    H5O__ainfo_decode,         
    H5O__ainfo_encode,         
    H5O__ainfo_copy,           
    H5O__ainfo_size,           
    NULL,                      
    H5O__ainfo_free,           
    H5O__ainfo_delete,         
    NULL,                      
    NULL,                      
    NULL,                      
    H5O__ainfo_pre_copy_file,  
    H5O__ainfo_copy_file,      
    H5O__ainfo_post_copy_file, 
    NULL,                      
    NULL,                      
    H5O__ainfo_debug           
}};

#define H5O_AINFO_VERSION 0

#define H5O_AINFO_TRACK_CORDER 0x01
#define H5O_AINFO_INDEX_CORDER 0x02
#define H5O_AINFO_ALL_FLAGS    (H5O_AINFO_TRACK_CORDER | H5O_AINFO_INDEX_CORDER)

H5FL_DEFINE_STATIC(H5O_ainfo_t);

static void *
H5O__ainfo_decode(H5F_t *f, H5O_t H5_ATTR_UNUSED *open_oh, unsigned H5_ATTR_UNUSED mesg_flags,
                  unsigned H5_ATTR_UNUSED *ioflags, size_t p_size, const uint8_t *p)
{
    const uint8_t *p_end = p + p_size - 1; 
    H5O_ainfo_t   *ainfo = NULL;           
    unsigned char  flags;                  
    uint8_t        sizeof_addr;            
    void          *ret_value = NULL;       

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    sizeof_addr = H5F_sizeof_addr(f);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_AINFO_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (ainfo = H5FL_MALLOC(H5O_ainfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    flags = *p++;
    if (flags & ~H5O_AINFO_ALL_FLAGS)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad flag value for message");
    ainfo->track_corder = (flags & H5O_AINFO_TRACK_CORDER) ? true : false;
    ainfo->index_corder = (flags & H5O_AINFO_INDEX_CORDER) ? true : false;

    
    ainfo->nattrs = HSIZET_MAX;

    
    if (ainfo->track_corder) {
        if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        UINT16DECODE(p, ainfo->max_crt_idx);
    }
    else
        ainfo->max_crt_idx = H5O_MAX_CRT_ORDER_IDX;

    
    H5_WARN_USELESS_COMPARISON_OFF
    if (H5_IS_BUFFER_OVERFLOW(p, sizeof_addr, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5_WARN_USELESS_COMPARISON_ON
    H5F_addr_decode(f, &p, &(ainfo->fheap_addr));

    
    H5_WARN_USELESS_COMPARISON_OFF
    if (H5_IS_BUFFER_OVERFLOW(p, sizeof_addr, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    H5_WARN_USELESS_COMPARISON_ON
    H5F_addr_decode(f, &p, &(ainfo->name_bt2_addr));

    
    if (ainfo->index_corder) {
        H5_WARN_USELESS_COMPARISON_OFF
        if (H5_IS_BUFFER_OVERFLOW(p, sizeof_addr, p_end))
            HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
        H5_WARN_USELESS_COMPARISON_ON
        H5F_addr_decode(f, &p, &(ainfo->corder_bt2_addr));
    }
    else
        ainfo->corder_bt2_addr = HADDR_UNDEF;

    
    ret_value = ainfo;

done:
    if (ret_value == NULL && ainfo != NULL)
        ainfo = H5FL_FREE(H5O_ainfo_t, ainfo);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ainfo_encode(H5F_t *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                  const void *_mesg)
{
    const H5O_ainfo_t *ainfo = (const H5O_ainfo_t *)_mesg;
    unsigned char      flags; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(ainfo);

    
    *p++ = H5O_AINFO_VERSION;

    
    flags = (unsigned char)(ainfo->track_corder ? H5O_AINFO_TRACK_CORDER : 0);
    flags = (unsigned char)(flags | (ainfo->index_corder ? H5O_AINFO_INDEX_CORDER : 0));
    *p++  = flags;

    
    if (ainfo->track_corder)
        UINT16ENCODE(p, ainfo->max_crt_idx);

    
    H5F_addr_encode(f, &p, ainfo->fheap_addr);

    
    H5F_addr_encode(f, &p, ainfo->name_bt2_addr);

    
    if (ainfo->index_corder)
        H5F_addr_encode(f, &p, ainfo->corder_bt2_addr);
    else
        assert(!H5_addr_defined(ainfo->corder_bt2_addr));

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__ainfo_copy(const void *_mesg, void *_dest)
{
    const H5O_ainfo_t *ainfo     = (const H5O_ainfo_t *)_mesg;
    H5O_ainfo_t       *dest      = (H5O_ainfo_t *)_dest;
    void              *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(ainfo);
    if (!dest && NULL == (dest = H5FL_MALLOC(H5O_ainfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *ainfo;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__ainfo_size(const H5F_t *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_ainfo_t *ainfo     = (const H5O_ainfo_t *)_mesg;
    size_t             ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    ret_value =
        (size_t)(1                               
                 + 1                             
                 + (ainfo->track_corder ? 2 : 0) 
                 + H5F_SIZEOF_ADDR(f)            
                 + H5F_SIZEOF_ADDR(f)            
                 + (ainfo->index_corder
                        ? H5F_SIZEOF_ADDR(f)
                        : 0)); 

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ainfo_free(void *mesg)
{
    FUNC_ENTER_PACKAGE_NOERR

    assert(mesg);

    mesg = H5FL_FREE(H5O_ainfo_t, mesg);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__ainfo_delete(H5F_t *f, H5O_t H5_ATTR_NDEBUG_UNUSED *open_oh, void *_mesg)
{
    H5O_ainfo_t *ainfo     = (H5O_ainfo_t *)_mesg;
    herr_t       ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(ainfo);
    assert(open_oh);

    
    if (H5_addr_defined(ainfo->fheap_addr))
        
        if (H5A__dense_delete(f, ainfo) < 0)
            HGOTO_ERROR(H5E_OHDR, H5E_CANTFREE, FAIL, "unable to free dense attribute storage");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ainfo_pre_copy_file(H5F_t H5_ATTR_UNUSED *file_src, const void H5_ATTR_UNUSED *native_src, bool *deleted,
                         const H5O_copy_t *cpy_info, void H5_ATTR_UNUSED *udata)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(deleted);
    assert(cpy_info);

    
    if (cpy_info->copy_without_attr)
        *deleted = true;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__ainfo_copy_file(H5F_t H5_ATTR_NDEBUG_UNUSED *file_src, void *mesg_src, H5F_t *file_dst,
                     bool H5_ATTR_UNUSED *recompute_size, unsigned H5_ATTR_UNUSED *mesg_flags,
                     H5O_copy_t H5_ATTR_NDEBUG_UNUSED *cpy_info, void H5_ATTR_UNUSED *udata)
{
    H5O_ainfo_t *ainfo_src = (H5O_ainfo_t *)mesg_src;
    H5O_ainfo_t *ainfo_dst = NULL;
    void        *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(file_src);
    assert(ainfo_src);
    assert(file_dst);
    assert(cpy_info);
    assert(!cpy_info->copy_without_attr);

    
    if (NULL == (ainfo_dst = H5FL_MALLOC(H5O_ainfo_t)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *ainfo_dst = *ainfo_src;

    if (H5_addr_defined(ainfo_src->fheap_addr)) {
        

        
        H5_BEGIN_TAG(H5AC__COPIED_TAG)

        if (H5A__dense_create(file_dst, ainfo_dst) < 0)
            HGOTO_ERROR_TAG(H5E_OHDR, H5E_CANTINIT, NULL, "unable to create dense storage for attributes");

        
        H5_END_TAG
    } 

    
    ret_value = ainfo_dst;

done:
    
    if (!ret_value && ainfo_dst)
        ainfo_dst = H5FL_FREE(H5O_ainfo_t, ainfo_dst);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ainfo_post_copy_file(const H5O_loc_t *src_oloc, const void *mesg_src, H5O_loc_t *dst_oloc,
                          void *mesg_dst, unsigned H5_ATTR_UNUSED *mesg_flags, H5O_copy_t *cpy_info)
{
    const H5O_ainfo_t *ainfo_src = (const H5O_ainfo_t *)mesg_src;
    herr_t             ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    assert(ainfo_src);

    if (H5_addr_defined(ainfo_src->fheap_addr))
        if (H5A__dense_post_copy_file_all(src_oloc, ainfo_src, dst_oloc, (H5O_ainfo_t *)mesg_dst, cpy_info) <
            0)
            HGOTO_ERROR(H5E_ATTR, H5E_CANTCOPY, FAIL, "can't copy attribute");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__ainfo_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_ainfo_t *ainfo = (const H5O_ainfo_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(ainfo);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %" PRIuHSIZE "\n", indent, "", fwidth, "Number of attributes:", ainfo->nattrs);
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Track creation order of attributes:", ainfo->track_corder ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %s\n", indent, "", fwidth,
            "Index creation order of attributes:", ainfo->index_corder ? "TRUE" : "FALSE");
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Max. creation index value:", (unsigned)ainfo->max_crt_idx);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' attribute storage fractal heap address:", ainfo->fheap_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' attribute storage name index v2 B-tree address:", ainfo->name_bt2_addr);
    Rfprintf(stream, "%*s%-*s %" PRIuHADDR "\n", indent, "", fwidth,
            "'Dense' attribute storage creation order index v2 B-tree address:", ainfo->corder_bt2_addr);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
