/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5VLmodule.h" 

#include "H5private.h"  
#include "H5Eprivate.h" 
#include "H5Iprivate.h" 
#include "H5Pprivate.h" 
#include "H5Tprivate.h" 
#include "H5VLpkg.h"    

#include "H5VLnative_private.h" 

hid_t
H5VLregister_connector(const H5VL_class_t *cls, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (H5P_DEFAULT == vipl_id)
        vipl_id = H5P_VOL_INITIALIZE_DEFAULT;
    else if (true != H5P_isa_class(vipl_id, H5P_VOL_INITIALIZE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a VOL initialize property list");

    
    if (NULL == (connector = H5VL__register_connector_by_class(cls, vipl_id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL class");

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector");

done:
    if (ret_value < 0)
        
        if (connector && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID,
                        "unable to decrement ref count on VOL connector");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLregister_connector_by_name(const char *name, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!name)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID, "null VOL connector name is disallowed");
    if (0 == strlen(name))
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID,
                    "zero-length VOL connector name is disallowed");

    
    if (H5P_DEFAULT == vipl_id)
        vipl_id = H5P_VOL_INITIALIZE_DEFAULT;
    else if (true != H5P_isa_class(vipl_id, H5P_VOL_INITIALIZE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a VOL initialize property list");

    
    if (NULL == (connector = H5VL__register_connector_by_name(name, vipl_id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector");

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector ID");

done:
    if (ret_value < 0)
        
        if (connector && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID,
                        "unable to decrement ref count on VOL connector");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLregister_connector_by_value(H5VL_class_value_t value, hid_t vipl_id)
{
    H5VL_connector_t *connector = NULL;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (value < 0)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID,
                    "negative VOL connector value is disallowed");

    
    if (H5P_DEFAULT == vipl_id)
        vipl_id = H5P_VOL_INITIALIZE_DEFAULT;
    else if (true != H5P_isa_class(vipl_id, H5P_VOL_INITIALIZE))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "not a VOL initialize property list");

    
    if (NULL == (connector = H5VL__register_connector_by_value(value, vipl_id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector");

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector ID");

done:
    if (ret_value < 0)
        
        if (connector && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID,
                        "unable to decrement ref count on VOL connector");

    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5VLis_connector_registered_by_name(const char *name)
{
    htri_t ret_value = false; 

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5VL__is_connector_registered_by_name(name)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't check for VOL");

done:
    FUNC_LEAVE_API(ret_value)
} 

htri_t
H5VLis_connector_registered_by_value(H5VL_class_value_t connector_value)
{
    htri_t ret_value = false;

    FUNC_ENTER_API(FAIL)

    
    if ((ret_value = H5VL__is_connector_registered_by_value(connector_value)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't check for VOL");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLget_connector_id(hid_t obj_id)
{
    H5VL_object_t *vol_obj   = NULL;
    hid_t          ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, H5I_INVALID_HID, "invalid location identifier");

    
    if ((ret_value = H5VL_conn_register(vol_obj->connector)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "can't get VOL ID");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLget_connector_id_by_name(const char *name)
{
    H5VL_connector_t *connector = NULL;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (connector = H5VL__get_connector_by_name(name)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, H5I_INVALID_HID, "can't get VOL connector");

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector ID");

done:
    if (ret_value < 0)
        
        if (connector && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID,
                        "unable to decrement ref count on VOL connector");

    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLget_connector_id_by_value(H5VL_class_value_t connector_value)
{
    H5VL_connector_t *connector = NULL;
    hid_t             ret_value = H5I_INVALID_HID; 

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (NULL == (connector = H5VL__get_connector_by_value(connector_value)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, H5I_INVALID_HID, "can't get VOL connector");

    
    if ((ret_value = H5I_register(H5I_VOL, connector, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register VOL connector ID");

done:
    if (ret_value < 0)
        
        if (connector && H5VL_conn_dec_rc(connector) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID,
                        "unable to decrement ref count on VOL connector");

    FUNC_LEAVE_API(ret_value)
} 

ssize_t
H5VLget_connector_name(hid_t obj_id, char *name , size_t size)
{
    H5VL_object_t *vol_obj;
    ssize_t        ret_value = -1;

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "invalid VOL identifier");

    
    ret_value = (ssize_t)H5VL__get_connector_name(vol_obj->connector, name, size);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLclose(hid_t vol_id)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == H5I_object_verify(vol_id, H5I_VOL))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "not a VOL connector");

    
    if (H5I_dec_app_ref(vol_id) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to close VOL connector ID");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLunregister_connector(hid_t vol_id)
{
    H5VL_connector_t *native, *connector;
    int               cmp_value;           
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (connector = H5I_object_verify(vol_id, H5I_VOL)))
        HGOTO_ERROR(H5E_VOL, H5E_BADTYPE, FAIL, "not a VOL connector ID");

    
    native = H5VL_NATIVE_conn_g;
    if (H5VL_cmp_connector_cls(&cmp_value, connector->cls, native->cls) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");
    if (0 == cmp_value)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "unregistering the native VOL connector is not allowed");

    
    if (H5I_dec_app_ref(vol_id) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, FAIL, "unable to unregister VOL connector");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLcmp_connector_cls(int *cmp, hid_t connector_id1, hid_t connector_id2)
{
    H5VL_connector_t *conn1, *conn2;       
    herr_t            ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == (conn1 = H5I_object_verify(connector_id1, H5I_VOL)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a VOL connector ID");
    if (NULL == (conn2 = H5I_object_verify(connector_id2, H5I_VOL)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a VOL connector ID");

    
    if (H5VL_cmp_connector_cls(cmp, conn1->cls, conn2->cls) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTCOMPARE, FAIL, "can't compare connector classes");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLwrap_register(void *obj, H5I_type_t type)
{
    hid_t ret_value; 

    
    FUNC_ENTER_API_NOINIT

    
    
    switch (type) {
        case H5I_FILE:
        case H5I_GROUP:
        case H5I_DATATYPE:
        case H5I_DATASET:
        case H5I_MAP:
        case H5I_ATTR:
            
            break;
        case H5I_UNINIT:
        case H5I_BADID:
        case H5I_DATASPACE:
        case H5I_VFL:
        case H5I_VOL:
        case H5I_GENPROP_CLS:
        case H5I_GENPROP_LST:
        case H5I_ERROR_CLASS:
        case H5I_ERROR_MSG:
        case H5I_ERROR_STACK:
        case H5I_SPACE_SEL_ITER:
        case H5I_EVENTSET:
        case H5I_NTYPES:
        default:
            HGOTO_ERROR(H5E_VOL, H5E_BADRANGE, H5I_INVALID_HID, "invalid type number");
    } 
    if (NULL == obj)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, H5I_INVALID_HID, "obj is NULL");

    
    if ((ret_value = H5VL_wrap_register(type, obj, true)) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to wrap object");

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

void *
H5VLobject(hid_t id)
{
    void *ret_value; 

    FUNC_ENTER_API(NULL)

    
    if (NULL == (ret_value = H5VL_object(id)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, NULL, "unable to retrieve object");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLobject_is_native(hid_t obj_id, bool *is_native)
{
    H5VL_object_t *vol_obj   = NULL;
    herr_t         ret_value = SUCCEED;

    FUNC_ENTER_API(FAIL)

    if (!is_native)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "`is_native` argument is NULL");

    
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    if (H5VL_object_is_native(vol_obj, is_native) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't determine if object is a native connector object");

done:
    FUNC_LEAVE_API(ret_value)
} 

hid_t
H5VLget_file_type(void *file_obj, hid_t connector_id, hid_t dtype_id)
{
    H5T_t            *dtype;               
    H5T_t            *file_type    = NULL; 
    hid_t             file_type_id = -1;   
    H5VL_connector_t *connector;           
    H5VL_object_t    *file_vol_obj = NULL; 
    hid_t             ret_value    = -1;   

    FUNC_ENTER_API(H5I_INVALID_HID)

    
    if (!file_obj)
        HGOTO_ERROR(H5E_ARGS, H5E_UNINITIALIZED, H5I_INVALID_HID, "no file object supplied");
    if (NULL == (dtype = H5I_object_verify(dtype_id, H5I_DATATYPE)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a data type");
    if (NULL == (connector = H5I_object_verify(connector_id, H5I_VOL)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "not a file VOL ID");

    
    if (H5T_GET_FORCE_CONV(dtype))
        if (NULL == (file_vol_obj = H5VL_new_vol_obj(H5I_FILE, file_obj, connector, true)))
            HGOTO_ERROR(H5E_VOL, H5E_CANTCREATE, FAIL, "can't create VOL object");

    
    if (NULL == (file_type = H5T_copy(dtype, H5T_COPY_TRANSIENT)))
        HGOTO_ERROR(H5E_VOL, H5E_CANTCOPY, H5I_INVALID_HID, "unable to copy datatype");

    
    if ((file_type_id = H5I_register(H5I_DATATYPE, file_type, false)) < 0) {
        (void)H5T_close_real(file_type);
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, H5I_INVALID_HID, "unable to register file datatype");
    } 

    
    if (H5T_set_loc(file_type, file_vol_obj, H5T_LOC_DISK) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTINIT, H5I_INVALID_HID, "can't set datatype location");

    
    if (file_vol_obj) {
        if (H5VL_free_object(file_vol_obj) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID, "unable to free VOL object");
        file_vol_obj = NULL;
    } 

    
    ret_value = file_type_id;

done:
    
    if (ret_value < 0) {
        if (file_vol_obj && H5VL_free_object(file_vol_obj) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID, "unable to free VOL object");
        if (file_type_id >= 0 && H5I_dec_ref(file_type_id) < 0)
            HDONE_ERROR(H5E_VOL, H5E_CANTDEC, H5I_INVALID_HID, "unable to close file datatype");
    } 

    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLretrieve_lib_state(void **state )
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOINIT

    
    if (NULL == state)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "invalid state pointer");

    
    if (H5VL_retrieve_lib_state(state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "can't retrieve library state");

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5VLopen_lib_context(void **context)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOINIT

    
    if (NULL == context)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "invalid context pointer");

    
    if (H5VL_start_lib_state(context) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't start new library state");

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5VLrestore_lib_state(const void *state)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOINIT

    
    if (NULL == state)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "invalid state pointer");

    
    if (H5VL_restore_lib_state(state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTSET, FAIL, "can't restore library state");

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5VLclose_lib_context(void *context)
{
    herr_t ret_value = SUCCEED; 

    
    FUNC_ENTER_API_NOINIT

    
    if (NULL == context)
        HGOTO_ERROR(H5E_VOL, H5E_BADVALUE, FAIL, "invalid context pointer");

    
    if (H5VL_finish_lib_state(context) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTRESET, FAIL, "can't reset library state");

done:
    FUNC_LEAVE_API_NOINIT(ret_value)
} 

herr_t
H5VLfree_lib_state(void *state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == state)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid state pointer");

    
    if (H5VL_free_lib_state(state) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTRELEASE, FAIL, "can't free library state");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLquery_optional(hid_t obj_id, H5VL_subclass_t subcls, int opt_type, uint64_t *flags )
{
    H5VL_object_t *vol_obj   = NULL;
    herr_t         ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == flags)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid 'flags' pointer");
    if (NULL == (vol_obj = H5VL_vol_object(obj_id)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADTYPE, FAIL, "invalid object identifier");

    
    if (H5VL_introspect_opt_query(vol_obj, subcls, opt_type, flags) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTGET, FAIL, "unable to query VOL connector operation");

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLregister_opt_operation(H5VL_subclass_t subcls, const char *op_name, int *op_val )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == op_val)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_val pointer");
    if (NULL == op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name pointer");
    if ('\0' == *op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name string");
    if (!((H5VL_SUBCLS_ATTR == subcls) || (H5VL_SUBCLS_DATASET == subcls) ||
          (H5VL_SUBCLS_DATATYPE == subcls) || (H5VL_SUBCLS_FILE == subcls) || (H5VL_SUBCLS_GROUP == subcls) ||
          (H5VL_SUBCLS_OBJECT == subcls) || (H5VL_SUBCLS_LINK == subcls) || (H5VL_SUBCLS_REQUEST == subcls)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid VOL subclass type");

    
    if (H5VL__register_opt_operation(subcls, op_name, op_val) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREGISTER, FAIL, "can't register dynamic optional operation: '%s'",
                    op_name);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLfind_opt_operation(H5VL_subclass_t subcls, const char *op_name, int *op_val )
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == op_val)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_val pointer");
    if (NULL == op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name pointer");
    if ('\0' == *op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name string");
    if (!((H5VL_SUBCLS_ATTR == subcls) || (H5VL_SUBCLS_DATASET == subcls) ||
          (H5VL_SUBCLS_DATATYPE == subcls) || (H5VL_SUBCLS_FILE == subcls) || (H5VL_SUBCLS_GROUP == subcls) ||
          (H5VL_SUBCLS_OBJECT == subcls) || (H5VL_SUBCLS_LINK == subcls) || (H5VL_SUBCLS_REQUEST == subcls)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid VOL subclass type");

    
    if (H5VL__find_opt_operation(subcls, op_name, op_val) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_NOTFOUND, FAIL, "can't find dynamic optional operation: '%s'", op_name);

done:
    FUNC_LEAVE_API(ret_value)
} 

herr_t
H5VLunregister_opt_operation(H5VL_subclass_t subcls, const char *op_name)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_API(FAIL)

    
    if (NULL == op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name pointer");
    if ('\0' == *op_name)
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid op_name string");
    if (!((H5VL_SUBCLS_ATTR == subcls) || (H5VL_SUBCLS_DATASET == subcls) ||
          (H5VL_SUBCLS_DATATYPE == subcls) || (H5VL_SUBCLS_FILE == subcls) || (H5VL_SUBCLS_GROUP == subcls) ||
          (H5VL_SUBCLS_OBJECT == subcls) || (H5VL_SUBCLS_LINK == subcls) || (H5VL_SUBCLS_REQUEST == subcls)))
        HGOTO_ERROR(H5E_ARGS, H5E_BADVALUE, FAIL, "invalid VOL subclass type");

    
    if (H5VL__unregister_opt_operation(subcls, op_name) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_CANTREMOVE, FAIL, "can't unregister dynamic optional operation: '%s'",
                    op_name);

done:
    FUNC_LEAVE_API(ret_value)
} 
