/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5Opkg.h"      
#include "H5MMprivate.h" 

static void  *H5O__btreek_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags,
                                 size_t p_size, const uint8_t *p);
static herr_t H5O__btreek_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                                 const void *_mesg);
static void  *H5O__btreek_copy(const void *_mesg, void *_dest);
static size_t H5O__btreek_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__btreek_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_BTREEK[1] = {{
    H5O_BTREEK_ID,          
    "v1 B-tree 'K' values", 
    sizeof(H5O_btreek_t),   
    0,                      
    H5O__btreek_decode,     
    H5O__btreek_encode,     
    H5O__btreek_copy,       
    H5O__btreek_size,       
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    NULL,                   
    H5O__btreek_debug       
}};

#define H5O_BTREEK_VERSION 0

static void *
H5O__btreek_decode(H5F_t H5_ATTR_NDEBUG_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                   unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                   const uint8_t *p)
{
    const uint8_t *p_end     = p + p_size - 1; 
    H5O_btreek_t  *mesg      = NULL;           
    void          *ret_value = NULL;           

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    
    if (H5_IS_BUFFER_OVERFLOW(p, 1, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    if (*p++ != H5O_BTREEK_VERSION)
        HGOTO_ERROR(H5E_OHDR, H5E_CANTLOAD, NULL, "bad version number for message");

    
    if (NULL == (mesg = (H5O_btreek_t *)H5MM_calloc(sizeof(H5O_btreek_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed for v1 B-tree 'K' message");

    
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->btree_k[H5B_CHUNK_ID]);
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->btree_k[H5B_SNODE_ID]);
    if (H5_IS_BUFFER_OVERFLOW(p, 2, p_end))
        HGOTO_ERROR(H5E_OHDR, H5E_OVERFLOW, NULL, "ran off end of input buffer while decoding");
    UINT16DECODE(p, mesg->sym_leaf_k);

    
    ret_value = (void *)mesg;

done:
    if (NULL == ret_value)
        H5MM_free(mesg);
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__btreek_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size,
                   uint8_t *p, const void *_mesg)
{
    const H5O_btreek_t *mesg = (const H5O_btreek_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg);

    
    *p++ = H5O_BTREEK_VERSION;
    UINT16ENCODE(p, mesg->btree_k[H5B_CHUNK_ID]);
    UINT16ENCODE(p, mesg->btree_k[H5B_SNODE_ID]);
    UINT16ENCODE(p, mesg->sym_leaf_k);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__btreek_copy(const void *_mesg, void *_dest)
{
    const H5O_btreek_t *mesg      = (const H5O_btreek_t *)_mesg;
    H5O_btreek_t       *dest      = (H5O_btreek_t *)_dest;
    void               *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    if (!dest && NULL == (dest = (H5O_btreek_t *)H5MM_malloc(sizeof(H5O_btreek_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL,
                    "memory allocation failed for shared message table message");

    
    *dest = *mesg;

    
    ret_value = dest;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__btreek_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared,
                 const void H5_ATTR_UNUSED *_mesg)
{
    size_t ret_value = 0;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);

    ret_value = 1 + 
                2 + 
                2 + 
                2;  

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__btreek_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_btreek_t *mesg = (const H5O_btreek_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Chunked storage internal B-tree 'K' value:", mesg->btree_k[H5B_CHUNK_ID]);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Symbol table node internal B-tree 'K' value:", mesg->btree_k[H5B_SNODE_ID]);
    Rfprintf(stream, "%*s%-*s %u\n", indent, "", fwidth,
            "Symbol table node leaf 'K' value:", mesg->sym_leaf_k);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
