% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/quality_assessment.R
\name{CFS_qa}
\alias{CFS_qa}
\title{Quality assessment: radius alignment}
\usage{
CFS_qa(
  dt.input,
  qa.label_data = "",
  qa.label_trt = "",
  qa.max_lag = 10,
  qa.max_iter = 100,
  qa.min_nseries = 100,
  qa.blcrit = 0.1,
  qa.mem_target = 0.6
)
}
\arguments{
\item{dt.input}{A data.table containing tree-ring measurements with required columns:
\describe{
  \item{species}{Character. Species identifier (must be single species)}
  \item{SampleID}{Character/Integer. Unique sample identifier}
  \item{Year}{Integer. Year of measurement}
  \item{RawRing}{Numeric. Raw ring-width measurement}
  \item{RW_trt}{Numeric. Treated/detrended ring-width series}
}}

\item{qa.label_data}{Character. Label identifier for the dataset (required)}

\item{qa.label_trt}{Character. Label identifier for the treatment method (required)}

\item{qa.max_lag}{Integer. Maximum lag for cross-correlation analysis (default: 10)}

\item{qa.max_iter}{Integer. Maximum iterations for chronologies refinement (default: 100)}

\item{qa.min_nseries}{Integer. Minimum number of series required (default: 100)}

\item{qa.blcrit}{Numeric. Borderline threshold criterion for quasi-pass classification (default: 0.1)}

\item{qa.mem_target}{Numeric. Proportion of free memory to use for batch processing
(0-1, default: 0.6). Higher values use more memory but may be faster.}
}
\value{
An object of class \code{cfs_qa} containing:
  \describe{
    \item{dt.ccf}{data.table with CCF results and QA codes (\code{qa_code}) for each series}
    \item{dt.chron}{data.table with chronologies statistics}
    \item{dt.stats}{data.table with summary statistics per radius}
    \item{dt.plots}{List of data.tables formatted for plotting (raw and treated series, CCF plots)}
    \item{qa.parms}{List of parameters used in the analysis}
  }
}
\description{
Performs comprehensive quality assurance analysis for tree-ring crossdating
using pairwise cross-correlation functions and iterative chronologies
refinement with automatic memory-efficient batch processing.
}
\details{
The function performs a two-step quality assurance process:

\strong{Step 1: Pairwise Cross-Correlation}
\itemize{
  \item Computes CCF for all pairs of treated series
  \item Uses auto-batching to manage memory efficiently
  \item Identifies initial qualified samples with max CCF at lag 0
}

\strong{Step 2: Iterative chronologies Refinement}
\itemize{
  \item Computes chronologies from qualified samples
  \item Evaluates each sample by running CCF with the chronologies
  \item Iteratively refines the qualified samples until convergence
}

\strong{QA Code Classification:}
\itemize{
  \item \code{pass}: Maximum correlation at lag 0
  \item \code{borderline}: Second highest correlation at lag 0 (within threshold)
  \item \code{pm1}: Maximum correlation at lag +/- 1 (slight misalignment)
  \item \code{highpeak}: Maximum at non-zero lag, >2x second highest
  \item \code{fail}: All other cases
}

\strong{Auto-batching:}
The function automatically determines optimal batch size based on:
\itemize{
  \item Available system memory
  \item Number of CPU cores
  \item Estimated memory per CCF operation
  \item Safety margins to prevent out-of-memory errors
}
}
\examples{
\donttest{
# loading processed data
dt.samples_trt <- readRDS(system.file("extdata", "dt.samples_trt.rds", package = "growthTrendR"))
# data processing
dt.samples_long <- prepare_samples_clim(dt.samples_trt, calbai = FALSE)
# rename to the reserved column name
data.table::setnames(dt.samples_long,
c("sample_id", "year", "rw_mm"), c("SampleID", "Year" ,"RawRing"))
# assign treated series
# users can decide their own treated series
# for rhub::rhub_check() on macos VECTOR_ELT issues
data.table::setorder(dt.samples_long, SampleID, Year)
dt.samples_long$RW_trt <-
  ave(
  as.numeric(dt.samples_long$RawRing),
  dt.samples_long$SampleID,
  FUN = function(x)
  if (length(x) > 1L) c(NA_real_, diff(x)) else NA_real_
  )
# quality check on radius alignment based on the treated series
dt.qa <-CFS_qa(dt.input = dt.samples_long, qa.label_data = "demo-samples",
qa.label_trt = "difference", qa.min_nseries = 5)
}

}
\seealso{
\code{\link{ccf}} for cross-correlation function
}
