% Generated by roxygen2: do not edit by hand
% Please edit documentation in
%   R/extract_most_likely_trait_values_for_focal_time.R
\name{extract_most_likely_trait_values_from_contMap_for_focal_time}
\alias{extract_most_likely_trait_values_from_contMap_for_focal_time}
\title{Extract continuous trait data mapped on a phylogeny at a given time in the past}
\usage{
extract_most_likely_trait_values_from_contMap_for_focal_time(
  contMap,
  ace = NULL,
  tip_data = NULL,
  focal_time,
  update_contMap = FALSE,
  keep_tip_labels = TRUE
)
}
\arguments{
\item{contMap}{Object of class \code{"contMap"}, typically generated with \code{\link[=prepare_trait_data]{prepare_trait_data()}}
or \code{\link[phytools:contMap]{phytools::contMap()}}, that contains a phylogenetic tree and associated continuous trait mapping.
The phylogenetic tree must be rooted and fully resolved/dichotomous,
but it does not need to be ultrametric (it can includes fossils).}

\item{ace}{Named numeric vector (Optional). Ancestral Character Estimates (ACE) of the internal nodes,
typically generated with \code{\link[phytools:fastAnc]{phytools::fastAnc()}}, \code{\link[phytools:anc.ML]{phytools::anc.ML()}}, or \code{\link[ape:ace]{ape::ace()}}.
Names are nodes_ID of the internal nodes. Values are ACE of the trait.
Needed to provide accurate estimates of trait values.}

\item{tip_data}{Named numeric vector (Optional). Tip values of the trait.
Names are nodes_ID of the internal nodes.
Needed to provide accurate tip values.}

\item{focal_time}{Integer. The time, in terms of time distance from the present,
at which the tree and mapping must be cut. It must be smaller than the root age of the phylogeny.}

\item{update_contMap}{Logical. Specify whether the mapped phylogeny (\code{contMap})
provided as input should be updated for visualization and returned among the outputs. Default is \code{FALSE}.
The update consists in cutting off branches and mapping that are younger than the \code{focal_time}.}

\item{keep_tip_labels}{Logical. Specify whether terminal branches with a single descendant tip
must retained their initial \code{tip.label} on the updated contMap. Default is \code{TRUE}.
Used only if \code{update_contMap = TRUE}.}
}
\value{
By default, the function returns a list with three elements.
\itemize{
\item \verb{$trait_data} A named numerical vector with ML trait values found along branches overlapping the \code{focal_time}. Names are the tip.label/tipward node ID.
\item \verb{$focal_time} Integer. The time, in terms of time distance from the present, at which the trait data were extracted.
\item \verb{$trait_data_type} Character string. Define the type of trait data as "continuous". Used in downstream analyses to select appropriate statistical processing.
}

If \code{update_contMap = TRUE}, the output is a list with four elements: \verb{$trait_data}, \verb{$focal_time}, \verb{$trait_data_type}, and \verb{$contMap}.
\itemize{
\item \verb{$contMap} An object of class that contains the updated \code{contMap} with  branches and mapping that are younger than the \code{focal_time} cut off.
The function also adds multiple useful sub-elements to the \verb{$contMap$tree} element.
\itemize{
\item \verb{$root_age} Integer. Stores the age of the root of the tree.
\item \verb{$nodes_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_node_ID} to the \code{initial_node_ID}. Each row is a node.
\item \verb{$initial_nodes_ID} Vector of character strings. Provides the initial ID of internal nodes. Used to plot internal node IDs as labels with \code{\link[ape:nodelabels]{ape::nodelabels()}}.
\item \verb{$edges_ID_df} Data.frame with two columns. Provides the conversion from the \code{new_edge_ID} to the \code{initial_edge_ID}. Each row is an edge/branch.
\item \verb{$initial_edges_ID} Vector of character strings. Provides the initial ID of edges/branches. Used to plot edge/branch IDs as labels with \code{\link[ape:nodelabels]{ape::edgelabels()}}.
}
}
}
\description{
Extracts the most likely trait values found along branches
at a specific time in the past (i.e. the \code{focal_time}).
Optionally, the function can update the mapped phylogeny (\code{contMap}) such as
branches overlapping the \code{focal_time} are shorten to the \code{focal_time}, and
the continuous trait mapping for the cut off branches are removed
by updating the \verb{$tree$maps} and \verb{$tree$mapped.edge} elements.
}
\details{
The mapped phylogeny (\code{contMap}) is cut at a specific time in the past
(i.e. the \code{focal_time}) and the current trait values of the overlapping edges/branches are extracted.

----- Extract \code{trait_data} -----

If providing only the \code{contMap} trait values at tips and internal nodes will be extracted from
the mapping of the \code{contMap} leading to a slight dependency with the actual tip data
and estimated ancestral character values.

True ML estimates will be used if \code{tip_data} and/or \code{ace} are provided as optional inputs.
In practice the discrepancy is negligible.

----- Update the \code{contMap} -----

To obtain an updated \code{contMap} alongside the trait data, set \code{update_contMap = TRUE}.
The update consists in cutting off branches and mapping that are younger than the \code{focal_time}.
\itemize{
\item When a branch with a single descendant tip is cut and \code{keep_tip_labels = TRUE},
the leaf left is labeled with the tip.label of the unique descendant tip.
\item When a branch with a single descendant tip is cut and \code{keep_tip_labels = FALSE},
the leaf left is labeled with the node ID of the unique descendant tip.
\item In all cases, when a branch with multiple descendant tips (i.e., a clade) is cut,
the leaf left is labeled with the node ID of the MRCA of the cut-off clade.
}

The continuous trait mapping \code{contMap} (\verb{$tree$maps} and \verb{$tree$mapped.edge}) is updated accordingly by removing mapping associated with the cut off branches.
}
\examples{
# ----- Example 1: Only extent taxa (Ultrametric tree) ----- #

## Prepare data

# Load eel data from the R package phytools
# Source: Collar et al., 2014; DOI: 10.1038/ncomms6505

library(phytools)
data(eel.tree)
data(eel.data)

# Extract body size
eel_data <- setNames(eel.data$Max_TL_cm,
                     rownames(eel.data))

# Get Ancestral Character Estimates based on a Brownian Motion model
# To obtain values at internal nodes
eel_ACE <- phytools::fastAnc(tree = eel.tree, x = eel_data)

# Run a Stochastic Mapping based on a Brownian Motion model
# to interpolate values along branches and obtain a "contMap" object
eel_contMap <- phytools::contMap(eel.tree, x = eel_data,
                                 res = 100, # Number of time steps
                                 plot = FALSE)

# Set focal time to 50 Mya
focal_time <- 50

\donttest{ # (May take several minutes to run)
## Extract trait data and update contMap for the given focal_time

# Extract from the contMap (values are not exact ML estimates)
eel_test <- extract_most_likely_trait_values_from_contMap_for_focal_time(
   contMap = eel_contMap,
   focal_time = focal_time,
   update_contMap = TRUE)
# Extract from tip data and ML estimates of ancestral characters (values are true ML estimates)
eel_test <- extract_most_likely_trait_values_from_contMap_for_focal_time(
   contMap = eel_contMap,
   ace = eel_ACE, tip_data = eel_data,
   focal_time = focal_time,
   update_contMap = TRUE)

## Visualize outputs

# Print trait data
eel_test$trait_data

# Plot node labels on initial stochastic map with cut-off
plot(eel_contMap, fsize = c(0.5, 1))
ape::nodelabels()
abline(v = max(phytools::nodeHeights(eel_contMap$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated contMap with initial node labels
plot(eel_test$contMap)
ape::nodelabels(text = eel_test$contMap$tree$initial_nodes_ID) }

# ----- Example 2: Include fossils (Non-ultrametric tree) ----- #

## Test with non-ultrametric trees like mammals in motmot

## Prepare data

# Load mammals phylogeny and data from the R package motmot included within deepSTRAPP
# Data source: Slater, 2013; DOI: 10.1111/2041-210X.12084
data("mammals", package = "deepSTRAPP")

mammals_tree <- mammals$mammal.phy
mammals_data <- setNames(object = mammals$mammal.mass$mean,
                         nm = row.names(mammals$mammal.mass))[mammals_tree$tip.label]

# Get Ancestral Character Estimates based on a Brownian Motion model
# To obtain values at internal nodes
mammals_ACE <- phytools::fastAnc(tree = mammals_tree, x = mammals_data)

# Run a Stochastic Mapping based on a Brownian Motion model
# to interpolate values along branches and obtain a "contMap" object
mammals_contMap <- phytools::contMap(mammals_tree, x = mammals_data,
                                     res = 100, # Number of time steps
                                     plot = FALSE)

# Set focal time to 80 Mya
focal_time <- 80

\donttest{ # (May take several minutes to run)
## Extract trait data and update contMap for the given focal_time

# Extract from the contMap (values are not exact ML estimates)
mammals_test <- extract_most_likely_trait_values_from_contMap_for_focal_time(
   contMap = mammals_contMap,
   focal_time = focal_time,
   update_contMap = TRUE)
# Extract from tip data and ML estimates of ancestral characters (values are true ML)
mammals_test <- extract_most_likely_trait_values_from_contMap_for_focal_time(
   contMap = mammals_contMap,
   ace = mammals_ACE, tip_data = mammals_data,
   focal_time = focal_time,
   update_contMap = TRUE)

## Visualize outputs

# Print trait data
mammals_test$trait_data

# Plot node labels on initial stochastic map with cut-off
phytools::plot.contMap(mammals_contMap, fsize = c(0.5, 1))
ape::nodelabels()
abline(v = max(phytools::nodeHeights(mammals_contMap$tree)[,2]) - focal_time,
       col = "red", lty = 2, lwd = 2)

# Plot updated contMap with initial node labels
phytools::plot.contMap(mammals_test$contMap, fsize = c(0.8, 1))
ape::nodelabels(text = mammals_test$contMap$tree$initial_nodes_ID) }

}
\seealso{
\code{\link[=cut_phylo_for_focal_time]{cut_phylo_for_focal_time()}} \code{\link[=cut_contMap_for_focal_time]{cut_contMap_for_focal_time()}}

Associated main function: \code{\link[=extract_most_likely_trait_values_for_focal_time]{extract_most_likely_trait_values_for_focal_time()}}

Sub-functions for other types of trait data:

\code{\link[=extract_most_likely_states_from_densityMaps_for_focal_time]{extract_most_likely_states_from_densityMaps_for_focal_time()}}
\code{\link[=extract_most_likely_ranges_from_densityMaps_for_focal_time]{extract_most_likely_ranges_from_densityMaps_for_focal_time()}}
}
\author{
Maël Doré
}
