% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/roots.R
\name{roots_newton}
\alias{roots_newton}
\title{Newton method for roots}
\usage{
roots_newton(
  f0,
  f1,
  x0 = 0,
  tol = 1e-09,
  imax = 1e+06,
  ftol = NULL,
  message = TRUE,
  logg = FALSE,
  ...
)
}
\arguments{
\item{f0}{A function of one variable. If the function includes variables,
these will have to be passed as additional variables, using the
same names in the function's definition (see examples).}

\item{f1}{A function equal to the first derivative of `f0`. Parameters that
are potentially included in `f0`, must be also included in `f1`.}

\item{x0}{A numeric variable. The initial guess starting Newton's algorithm.}

\item{tol}{A real small number. The smallest difference
between the new zero's approximation and the previous one, above 
which the algorithm keeps working. As soon as the difference is 
less than `tol`, the algorithm stops and the current approximation
is returned as the final approximation to the function's root.
Default value is 1e-9.}

\item{imax}{A positive integer. The maximum number of iterations of the
algorithm. The default value is 1e6, although convergence is 
normally obtained with a number of iterations much smaller than 
imax. imax is important to stop search in those cases in which 
the algorithm gets stuck in endless loops (non-convergence).}

\item{ftol}{A real small number. When `ftol` is not NULL (default value),
Newton's algorithm stops when \eqn{|f(x)| < ftol}. This
parameter essentially introduces a different stopping criterion.}

\item{message}{A logical variable to state whether messages about the 
root and the error have to be printed. The default
is for the messages to be printed (`message=TRUE`).}

\item{logg}{A logical variable to state whether information on the series
of approximating roots is printed (TRUE) or not (FALSE). Default
is for such information not to be printed (FALSE).}

\item{...}{Parameters passed to the two functions `f0` and `f1`, if any.}
}
\value{
A numeric value, the zero of the function (or, equivalently, the
        root of the equation \eqn{f(x)=0}).
}
\description{
Find the zero of a function of one variable, \eqn{f(x)}, given a starting
value close to the zero, using Newton method.
}
\details{
Finding the zero of \eqn{f(x)} is equivalent to finding the roots of the
equation:
\deqn{
 f(x) = 0 
}
The algorithm used is based on Newton method that needs an initial guess,
x0, and the analytic expression of the function's first derivative. The 
method has a much faster convergence rate than both the bisection and secant
methods, but it does not converge when the initial guess or any other
subsequent approximations accidentally coincide with an optimal point of the
function, i.e. a point at which the first derivative is zero. The algorithm
can also potentially be stuck in an endless loop of repeating values for
special combinations of functions and initial guess.
}
\examples{
# The quadratic equation x^2-5*x+6=0 has two roots, 2 and 3
f0 <- function(x) return(x^2-5*x+6)

# First derivative
f1 <- function(x) return(2*x-5)

# Find root 2, starting from a single point close to 2
x0 <- 1
x <- roots_newton(f0,f1,x0=1)
print(x)

# Find root 3 (no message printing)
x <- roots_newton(f0,f1,x0=4,message=FALSE)
print(x)

# Function with a parameter f(x) = exp(kx) - 2
f0 <- function(x,k=2) return(exp(k*x)-2)

# First derivative (it includes the parameter)
f1 <- function(x,k=2) return(k*exp(k*x))

# Solution of exp(2x)-2=0 is log(2)/2
x <- roots_newton(f0,f1,k=2)
print(log(2)/2)

}
