#' Plot 3D trajectories as variable-width lines
#'
#' Creates an interactive 3D plot showing research trajectories with time on the x-axis,
#' route separation on the y-axis, and cumulative paper counts on the z-axis. Highlighted
#' trajectories are displayed as growing-thickness lines, with optional background
#' trajectories and network context in lowlight style.
#'
#' @param traj_data List containing trajectory data generated by 
#'   `detect_main_trajectories()` with components:
#'   - `graph`: igraph object containing nodes and edges across years
#'   - `trajectories`: tibble of all candidate trajectories (traj_id + nodes list)
#' @param traj_filtered Filtered trajectories tibble from `filter_trajectories()` 
#'   containing the subset to emphasize. Must contain columns:
#'   - `traj_id`: trajectory identifiers
#'   - `nodes`: list of character vectors (ordered by time or orderable)
#' @param width_range_hi Width range for highlighted trajectory segments 
#'   (default: c(4, 12)). Segment widths scale with cumulative paper counts.
#' @param width_range_lo Baseline width range used to compute constant lowlight 
#'   width (default: c(1.2, 3)). The mean of this range determines lowlight width.
#' @param use_raw_papers Whether to use raw paper counts for z-axis scaling 
#'   (default: TRUE). If TRUE, uses raw `quantity_papers`; if FALSE, uses 
#'   weighted size: `quantity_papers * prop_tracked_intra_group`.
#' @param connect_only_existing_edges Whether to draw only edges that exist in 
#'   the graph (default: TRUE). If FALSE, draws all consecutive node pairs in 
#'   trajectories regardless of graph edges.
#' @param show_labels Whether to add end-of-trajectory labels inside the 3D plot 
#'   (default: TRUE)
#' @param show_only_highlighted Whether to show only highlighted trajectories 
#'   (default: FALSE). If TRUE, hides all background network and lowlight trajectories.
#' @param label_size Font size for trajectory end labels (default: 18)
#' @param hover_font_size Font size for hover tooltips (default: 12)
#' @param lowlight_width Line width for lowlight trajectories and background 
#'   network (default: 1)
#' @param lowlight_alpha Transparency for lowlight elements (default: 0.9)
#' @param lowlight_color Color for lowlight elements (default: "#9AA5B1" - neutral gray)
#'
#' @return A plotly interactive 3D plot object
#'
#' @details
#' This function creates an interactive 3D visualization of research trajectories:
#' - **X-axis**: Publication year (parsed from vertex names like "y2007g05")
#' - **Y-axis**: "Route" (Sugiyama layout coordinate to separate trajectories vertically)
#' - **Z-axis**: Cumulative documents (raw or weighted) along each trajectory
#' 
#' Key features:
#' - **Highlighted trajectories** (`traj_filtered`) are colored lines with widths 
#'   that grow proportionally to cumulative paper counts
#' - **Lowlight trajectories** (when `show_only_highlighted = FALSE`) show other 
#'   trajectories as constant-width lines
#' - **Background network** (when `show_only_highlighted = FALSE`) provides 
#'   context with thin gray edges
#' - **Hover tooltips** show detailed information at each trajectory point
#' - **End labels** identify highlighted trajectories (when `show_labels = TRUE`)
#' - **Edge validation** (when `connect_only_existing_edges = TRUE`) ensures only 
#'   actual graph edges are drawn
#'
#' The function uses a Sugiyama layout for the y-axis coordinates and cumulative
#' sums of paper counts for the z-axis values. Colors for highlighted trajectories
#' are assigned using RColorBrewer's Set2 palette (for <=8 trajectories) or a 
#' hue-based palette (for more trajectories).
#'
#' @examples
#' \dontrun{
#' # Detect main trajectories first
#' traj_data <- detect_main_trajectories(your_graph_data)
#' 
#' # Filter trajectories of interest
#' filtered_traj <- filter_trajectories(traj_data$trajectories, 
#'                                      min_papers = 10)
#' 
#' # Create interactive 3D plot
#' plot_group_trajectories_lines_3d(
#'   traj_data = traj_data,
#'   traj_filtered = filtered_traj,
#'   width_range_hi = c(3, 10),
#'   use_raw_papers = FALSE,
#'   show_labels = TRUE
#' )
#' 
#' # Minimal view with only highlighted trajectories
#' plot_group_trajectories_lines_3d(
#'   traj_data = traj_data,
#'   traj_filtered = filtered_traj,
#'   show_only_highlighted = TRUE,
#'   label_size = 16
#' )
#' }
#'
#' @export
#' @importFrom igraph vcount V E as_edgelist get.edge.ids
#' @importFrom plotly plot_ly add_trace layout
#' @importFrom RColorBrewer brewer.pal
#' @importFrom scales hue_pal alpha rescale
#' @importFrom tidygraph as_tbl_graph
#' @importFrom ggraph create_layout
#' @importFrom stats setNames
plot_group_trajectories_lines_3d <- function(
  traj_data,
  traj_filtered,
  width_range_hi = c(4, 12),
  width_range_lo = c(1.2, 3),
  use_raw_papers = TRUE,
  connect_only_existing_edges = TRUE,
  show_labels = TRUE,
  show_only_highlighted = FALSE,
  label_size = 18,
  hover_font_size = 12,
  lowlight_width = 1,
  lowlight_alpha = 0.9,
  lowlight_color = "#9AA5B1"
) {
  if (!requireNamespace("plotly", quietly = TRUE))
    stop("plotly is required for 3D plotting. Please install.packages('plotly').", call. = FALSE)

  `%||%` <- function(a, b) if (!is.null(a)) a else b
  as_year <- function(x) as.integer(sub("^y(\\d{4}).*$", "\\1", x))

  # Input validation
  if (!is.list(traj_data) || !all(c("graph", "trajectories") %in% names(traj_data))) {
    stop("traj_data must be a list with 'graph' and 'trajectories' components")
  }
  
  g <- traj_data$graph
  tr_all <- traj_data$trajectories
  tr_highlight <- traj_filtered
  
  stopifnot(inherits(g, "igraph"))
  if (igraph::vcount(g) == 0) return(plotly::plot_ly())

  # Layout (Sugiyama vertical coord -> use layout X as 'Route' axis)
  yrs <- as_year(igraph::V(g)$name)
  tg  <- tidygraph::as_tbl_graph(g)
  lay <- ggraph::create_layout(tg, layout = "sugiyama", layers = yrs)
  v_y <- setNames(lay$x, lay$name)

  # Node measure for Z
  node_measure <- if (use_raw_papers) {
    (igraph::V(g)$quantity_papers %||% 0)
  } else {
    (igraph::V(g)$quantity_papers %||% 0) * (igraph::V(g)$prop_tracked_intra_group %||% 0)
  }
  nm <- igraph::V(g)$name

  # Initialize plot
  p <- plotly::plot_ly()

  # Background network (context) — draw ONLY if not highlights-only
  if (!show_only_highlighted) {
    el <- igraph::as_edgelist(g, names = TRUE)
    x_ctx <- y_ctx <- z_ctx <- numeric(0)
    for (i in seq_len(nrow(el))) {
      f <- el[i, 1]; t <- el[i, 2]
      x_ctx <- c(x_ctx, as_year(f), as_year(t), NA)
      y_ctx <- c(y_ctx, v_y[f],     v_y[t],     NA)
      z_ctx <- c(z_ctx, 0,          0,          NA)
    }
    p <- p |>
      plotly::add_trace(
        x = x_ctx, y = y_ctx, z = z_ctx,
        type = "scatter3d", mode = "lines",
        line = list(
          color = scales::alpha(lowlight_color, lowlight_alpha),
          width = max(0.5, lowlight_width)
        ),
        hoverinfo = "none", showlegend = FALSE
      )
  }

  # Build per-trajectory sequences
  prep_trajs <- function(tr_tbl) {
    if (is.null(tr_tbl) || !nrow(tr_tbl)) return(list())
    out <- vector("list", nrow(tr_tbl))
    for (i in seq_len(nrow(tr_tbl))) {
      tid   <- tr_tbl$traj_id[i]
      nodes <- tr_tbl$nodes[[i]]
      if (length(nodes) < 2) next
      ord   <- order(as_year(nodes), nodes); nodes <- nodes[ord]
      years <- as_year(nodes)
      yvals <- v_y[nodes]
      step_p <- node_measure[match(nodes, nm)]; step_p[is.na(step_p)] <- 0
      zvals <- cumsum(step_p)  # cumulative
      out[[i]] <- list(traj_id = tid, nodes = nodes, years = years,
                       yvals = yvals, step_p = step_p, zvals = zvals)
    }
    Filter(Negate(is.null), out)
  }
  td_all <- prep_trajs(tr_all)
  td_hi  <- prep_trajs(tr_highlight)

  # Colors for highlights
  if (length(td_hi) <= 8) pal_hi <- RColorBrewer::brewer.pal(max(3, length(td_hi)), "Set2")[seq_along(td_hi)]
  else                    pal_hi <- scales::hue_pal()(length(td_hi))
  col_map_hi <- if (length(td_hi)) setNames(pal_hi, vapply(td_hi, `[[`, "", "traj_id")) else character(0)

  # Lowlight set
  td_lo <- if (show_only_highlighted) list() else td_all
  lw_lo <- mean(width_range_lo)

  # Draw LOWLIGHT trajectories (constant thin lines) — only if not highlights-only
  if (length(td_lo)) {
    for (td in td_lo) {
      nodes <- td$nodes; years <- td$years; yvals <- td$yvals; zvals <- td$zvals
      if (!connect_only_existing_edges) {
        p <- p |>
          plotly::add_trace(
            x = years, y = yvals, z = zvals,
            type = "scatter3d", mode = "lines",
            line = list(
              color = scales::alpha(lowlight_color, lowlight_alpha),
              width = max(lowlight_width, lw_lo)
            ),
            name = td$traj_id, showlegend = FALSE, hoverinfo = "skip"
          )
      } else {
        xs <- ys <- zs <- c()
        for (i in seq_len(length(nodes) - 1)) {
          eid <- igraph::get.edge.ids(g, c(nodes[i], nodes[i + 1]))
          if (eid != 0) {
            xs <- c(xs, years[i], years[i + 1], NA)
            ys <- c(ys, yvals[i], yvals[i + 1], NA)
            zs <- c(zs, zvals[i], zvals[i + 1], NA)
          }
        }
        if (length(xs)) {
          p <- p |>
            plotly::add_trace(
              x = xs, y = ys, z = zs,
              type = "scatter3d", mode = "lines",
              line = list(
                color = scales::alpha(lowlight_color, lowlight_alpha),
                width = max(lowlight_width, lw_lo)
              ),
              name = td$traj_id, showlegend = FALSE, hoverinfo = "skip"
            )
        }
      }
    }
  }

  # Draw HIGHLIGHT trajectories (growing thickness)
  if (length(td_hi)) {
    for (td in td_hi) {
      col <- unname(col_map_hi[td$traj_id])

      seg_widths <- if (length(unique(td$zvals)) <= 1) {
        rep(mean(width_range_hi), max(0, length(td$nodes) - 1))
      } else {
        scales::rescale(td$zvals[-1], to = width_range_hi, from = range(td$zvals, na.rm = TRUE))
      }

      for (i in seq_len(length(td$nodes) - 1)) {
        if (connect_only_existing_edges) {
          eid <- igraph::get.edge.ids(g, c(td$nodes[i], td$nodes[i + 1]))
          if (eid == 0) next
        }
        p <- p |>
          plotly::add_trace(
            x = c(td$years[i], td$years[i + 1]),
            y = c(td$yvals[i], td$yvals[i + 1]),
            z = c(td$zvals[i], td$zvals[i + 1]),
            type = "scatter3d", mode = "lines",
            line = list(color = col, width = seg_widths[i]),
            name = td$traj_id, showlegend = (i == 1),
            hoverinfo = "skip"
          )
      }

      # Invisible markers for hover
      hover_txt <- paste0(
        "<b>", td$traj_id, "</b><br>",
        "Year: ", td$years, "<br>",
        "Step papers: ", td$step_p, "<br>",
        "Cumulative: ", td$zvals
      )
      p <- p |>
        plotly::add_trace(
          x = td$years, y = td$yvals, z = td$zvals,
          type = "scatter3d", mode = "markers",
          marker = list(size = 1.5, opacity = 0),
          hoverinfo = "text", text = hover_txt,
          hoverlabel = list(
            bgcolor = "rgba(50,50,50,0.9)",
            font = list(color = "white", size = hover_font_size)
          ),
          showlegend = FALSE
        )

      if (show_labels) {
        p <- p |>
          plotly::add_trace(
            x = tail(td$years, 1), y = tail(td$yvals, 1), z = tail(td$zvals, 1),
            type = "scatter3d", mode = "text",
            text = td$traj_id,
            textfont = list(size = label_size, color = col),
            showlegend = FALSE
          )
      }
    }
  }

  # Camera (match your other 3D view)
  camera_settings <- list(eye = list(x = -1.5, y = -2, z = 1))

  p |>
    plotly::layout(
      scene = list(
        xaxis = list(title = "Year"),
        yaxis = list(title = "Route"),
        zaxis = list(title = if (use_raw_papers) "Cumulative documents" else "Cumulative weighted documents"),
        camera = camera_settings
      ),
      legend = list(font = list(size = 16)),
      title = "3D Trajectories"
    )
}
