#' Load a BERTopic Model Bundle
#'
#' Load a BERTopic model saved with `save_bertopic_model()` along with its
#' companion RDS file containing R-side extras.
#'
#' @param path Directory path where the Python model was saved.
#' @param embedding_model Optional embedding model to pass through to
#'   `BERTopic$load()` when the embedding model is not serialized.
#' @return A list with two elements: `model` (the BERTopic model) and `extras`
#'   (the R-side data saved in the companion RDS file).
#' @export
#' @examples
#' \\donttest{
#' if (dir.exists("topic_model")) {
#'   loaded <- load_bertopic_model("topic_model")
#'   print(loaded$extras)
#' } else {
#'   message("No saved model found at 'topic_model'.")
#' }
#' }
load_bertopic_model <- function(path, embedding_model = NULL) {
  bertopic <- reticulate::import("bertopic")

  if (is.null(embedding_model)) {
    model <- bertopic$BERTopic$load(path)
  } else {
    model <- bertopic$BERTopic$load(path, embedding_model = embedding_model)
  }

  extras <- readRDS(paste0(path, "_extras.rds"))
  list(model = model, extras = extras)
}

