% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/col2Munsell.R
\name{col2Munsell}
\alias{col2Munsell}
\title{Convert colors into Munsell Notation}
\usage{
col2Munsell(col, space = c("sRGB", "CIELAB"), nClosest = 1)
}
\arguments{
\item{col}{character vector of colors, \code{data.frame} or \code{matrix} of color coordinates in sRGB or CIELAB color space}

\item{space}{character, one of \code{sRGB} or \code{CIELAB}, defines the input color system}

\item{nClosest}{integer, number of closest Munsell colors to return (valid range is 1-20)}
}
\value{
an (NA-padded) \code{data.frame} containing \code{hue}, \code{value}, \code{chroma}, and CIE delta-E 2000 color contrast metric between source and nearest matching color(s).
}
\description{
Lookup the \code{n} closest Munsell chips from the \code{munsell} lookup table from various color notations. This function replaces \code{rgb2munsell()}.
}
\note{
This function is fully vectorized and will pad output with NA-records when NA are present in \code{color}.
}
\examples{

# vector of named R colors
col2Munsell(c('red', 'green', 'blue'))

# sRGB matrix in the range of 0-255
col2Munsell(cbind(255, 0, 0))

# sRGB matrix in the range of 0-1
col2Munsell(cbind(1, 0, 0))

# 10YR 5/6 in CIELAB
col2Munsell(
  cbind(51.4337, 9.917916, 38.6889), 
  space = 'CIELAB'
)

# 2.5YR 6/8 in hex notation
col2Munsell("#D18158FF")

# 7.5YR 8/1 in sRGB {0, 1}
col2Munsell(
  cbind(0.8240707, 0.7856834, 0.7541048)
)

# 7.5YR 8/1 in sRGB {0, 255}
col2Munsell(
  cbind(0.8240707, 0.7856834, 0.7541048) * 255
)

# multple colors in CIELAB
col2Munsell(
  parseMunsell(c('10BG 6/6', '2.5YR 4/6'), returnLAB = TRUE),
  space = 'CIELAB'
)

# data.frame input
col2Munsell(
  data.frame(r = 1, g = 0, b = 0),
  space = 'sRGB'
)

# keep examples from using more than 2 cores
data.table::setDTthreads(Sys.getenv("OMP_THREAD_LIMIT", unset = 2))

# Munsell notation to sRGB triplets {0, 1} 
color <- munsell2rgb(
  the_hue = c('10YR', '2.5YR', '5YR'), 
  the_value = c(3, 5, 2.5), 
  the_chroma = c(5, 6, 2), 
  return_triplets = TRUE
)

# result is a data.frame of sRGB {0, 1}
color

# back-transform sRGB -> closest Munsell color
# sigma is the dE00 color contrast metric
col2Munsell(color, space = 'sRGB')

}
\references{
\itemize{
\item http://ncss-tech.github.io/AQP/
\item http://www.brucelindbloom.com/index.html?ColorCalcHelp.html
\item http://www.munsellcolourscienceforpainters.com/MunsellAndKubelkaMunkToolbox/MunsellAndKubelkaMunkToolbox.html
\item http://www.cis.rit.edu/mcsl/online/munsell.php
}
}
\author{
D.E. Beaudette
}
