% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/spp_trend.R
\name{spp_trend}
\alias{spp_trend}
\title{Individual trend analysis}
\usage{
spp_trend(data, spp, predictor, responses, n_min = 50)
}
\arguments{
\item{data}{A \verb{data frame} containing the variables for the model, including \code{species}, \code{year}, \code{month}, \code{lon}, \code{lat}, \code{tme} and/or \code{ele}.}

\item{spp}{A \code{character} vector of unique species names.}

\item{predictor}{A \code{character} vector of predictor variable names representing a temporal variable (\code{year_month}).}

\item{responses}{A \code{character} vector of response variable names to analyze.}

\item{n_min}{Minimum \code{numeric} number of presences required for a species in each hemisphere (or globally for species in both hemispheres) to perform the analysis.}
}
\value{
A data frame with trend statistics, including:
\itemize{
\item \code{species}: Name of the analyzed species.
\item \code{responses}: Name of the variable analyzed.
\item \code{trend}: Slope of the linear model (rate of change over time).
\item \code{t}: t-statistic for the species-specific trend.
\item \code{pvalue}: Statistical significance of the species trend.
\item \code{ci_95_max}, \code{ci_95_min}: 95\% confidence interval bounds for the slope.
\item \code{dif_t}: t-statistic of the interaction term (species vs. baseline).
\item \code{dif_pvalue}: p-values of the interaction term. A low value indicates a significant deviation from the general trend.
\item \code{n}: Sample size for the specific species/hemisphere subset
\item \code{hemisphere}: Geographic context (\code{North}, \code{South}, or \code{Global} for global comparison).
}
}
\description{
Estimates the species-specific temporal trends for each selected response variable and statistically compares them with the overall temporal trend derived from the complete dataset.
It compares individual species' trajectories against the OT using the interaction term of the \code{lm()}.
}
\details{
The function fits linear models for each species and compares them to the general trend using an interaction model (\code{response ~ predictor * group}).
Longitude (\code{lon}) values are transformed to a 0-360 range to ensure statistical consistency near the antimeridian.
A key feature of this function is its specialized handling of latitude. Because the Equator is set at 0, latitude values in the Southern Hemisphere are negative.
To ensure that a direction shift is interpreted consistently across the globe (where a negative increase in the South corresponds to a positive increase in the North), the function employs two complementary approaches:
Hemispheric split: It divides the records based on their location (\code{lat < 0} for \code{South} and \code{lat > 0} for \code{North}) and performs separate analyses for each.
Global analysis: It performs an analysis using the complete dataset (\code{Global}) by transforming all latitudes into absolute values (\code{abs(lat)}). This allows for a unified global trend estimation.
Note that this hemispheric division and absolute transformation logic is applied exclusively to the latitude (\code{lat}) variable.
}
\examples{

data <- data.frame(
   species = sample(paste0("spp_", 1:10), 500, replace = TRUE),
   year = sample(1950:2020, 500, replace = TRUE),
   month = sample(1:12, 500, replace = TRUE),
   lon = runif(500, -10, 20),
   lat = runif(500, 30, 70),
   tme = rnorm(500, 15, 10)
)

data$year_month <- data$year + data$month * 0.075

predictor <- "year_month"
responses <- c("lat", "lon", "tme")

spp <- unique(data$species)

spp_trend_result <- spp_trend(data, spp, predictor, responses, n_min = 50)

print(spp_trend_result)

}
