# In the paper 'Universal Resilience Patterns in Complex Networks' Barbasi describes several methods to
# reduce the dimension of differential equations of the type
#   dx/dt = f(beta, x)
#
# We implement these methods here


#' Average Nearest-Neighbor Activity
#'
#' This is the average nearest-neight activity described by Barabasi in his paper 'Universal resilience patterns in complex networks'. It can be used to characterize the effective state of the system.
#'
#'
#' @param M Adjacency matrix of the network that describes the system
#' @param x The value of all the nodes of the system
#'
#' @return A numeric value describing the effective state by the system by using the nearest-neighbour activity.
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    initial_parameters <- function(M){
#'      initial_params <- list('f' = 1, 'h' = 2, 'B' = 0.1, 'Beta' = Rato::Barabasi.B_eff(M))
#'      return(initial_params)
#'    }
#'    
#'    update_parameters <- function(params){
#'      M <- params$M
#'      params$Beta <- Rato::Barabasi.B_eff(M)
#'      return(params)
#'    }
#'  
#'    Rato::node.removal.thread( system = Rato::Michaelis.Menten.Fast
#'                             , M = g$M
#'                             , x0 = Rato::Barabasi.X_eff(g$M, g$initial_values)
#'                             , initial_params = initial_parameters
#'                             , update_params = update_parameters
#'                             , to.numeric = TRUE)
#'
#' }
Barabasi.X_eff <- function(M, x) {
  indegree <- colSums(M)
  outdegree <- rowSums(M)
  n <- nrow(M)
  ones <- matrix(1, nrow = n, ncol = 1)

  x_eff <- (outdegree * x) / sum(indegree)

  num <- as.numeric(t(ones) %*% M %*% x)
  den <- as.numeric(t(ones) %*% M %*% ones)

  return(num / den)
}

#' Average the product of the outgoing and incoming degrees of nodes
#'
#' This is the average of the outgoing/incoming nodes, described by Barabasi in his paper 'Universal resilience patterns in complex networks'.
#'
#'
#' @param M Adjacency matrix of the network that describes the system
#'
#' @return A numeric value describing the Beta_eff described by Barabasi.
#' @export
#' @examples
#' \donttest{
#'    node_file <- system.file("extdata", "IL17.nodes.csv", package = "Rato")
#'    edge_file <- system.file("extdata", "IL17.edges.csv", package = "Rato")
#'    g <- Rato::graph.from.csv(node_file, edge_file, sep=",", header=TRUE)
#'  
#'    initial_parameters <- function(M){
#'      initial_params <- list('f' = 1, 'h' = 2, 'B' = 0.1, 'Beta' = Rato::Barabasi.B_eff(M))
#'      return(initial_params)
#'    }
#'    
#'    update_parameters <- function(params){
#'      M <- params$M
#'      params$Beta <- Rato::Barabasi.B_eff(M)
#'      return(params)
#'    }
#'  
#'    Rato::node.removal.thread( system = Rato::Michaelis.Menten.Fast
#'                             , M = g$M
#'                             , x0 = Rato::Barabasi.X_eff(g$M, g$initial_values)
#'                             , initial_params = initial_parameters
#'                             , update_params = update_parameters
#'                             , to.numeric = TRUE)
#'
#' }
Barabasi.B_eff <- function(M){
  indegree <- colSums(M)
  outdegree <- rowSums(M)

  numerator <- (indegree %*% outdegree)
  denominator <- sum(indegree)

  if(numerator == 0 && denominator == 0){
    Beta <- 0
  } else {
    Beta <- ((indegree %*% outdegree) / sum(indegree))
  }

  return(Beta)
}
