#' @title Quantile-on-Quantile Regression
#'
#' @description
#' Performs Quantile-on-Quantile (QQ) regression analysis as described in 
#' Sim and Zhou (2015). This approach estimates the effect that quantiles of 
#' an independent variable have on quantiles of a dependent variable, capturing 
#' the dependence between their distributions.
#'
#' @param y Numeric vector. The dependent variable.
#' @param x Numeric vector. The independent variable.
#' @param y_quantiles Numeric vector. Quantiles of the dependent variable to 
#'   estimate. Default is \code{seq(0.05, 0.95, by = 0.05)}.
#' @param x_quantiles Numeric vector. Quantiles of the independent variable 
#'   to condition on. Default is \code{seq(0.05, 0.95, by = 0.05)}.
#' @param min_obs Integer. Minimum number of observations required for 
#'   quantile regression. Default is 10.
#' @param se_method Character string. Method for computing standard errors 
#'   in quantile regression. One of "boot" (bootstrap), "nid" (no i.i.d. 
#'   assumption), "iid", or "ker" (kernel). Default is "boot".
#' @param verbose Logical. If \code{TRUE}, prints progress messages. 
#'   Default is \code{TRUE}.
#'
#' @return An object of class "qq_regression" containing:
#' \itemize{
#'   \item \code{results} - Data frame with columns: y_quantile, x_quantile, 
#'     coefficient, std_error, t_value, p_value, r_squared
#'   \item \code{y_quantiles} - Quantiles of y used
#'   \item \code{x_quantiles} - Quantiles of x used
#'   \item \code{n_obs} - Number of complete observations
#'   \item \code{call} - The matched call
#'   \item \code{method} - Method used ("quantile_regression" or "correlation")
#' }
#'
#' @details
#' The QQ approach combines quantile regression with local linear regression 
#' to estimate the effect of the tau-quantile of x on the theta-quantile of y.
#' 
#' The method proceeds as follows:
#' \enumerate{
#'   \item For each x_quantile (tau), subset the data where x <= quantile(x, tau)
#'   \item For each y_quantile (theta), perform quantile regression of y on x
#'   \item Extract coefficients, standard errors, t-values, p-values
#'   \item Compute pseudo R-squared based on check function
#' }
#'
#' If the subset has fewer than \code{min_obs} observations, a correlation-based
#' approach is used instead.
#'
#' @references
#' Sim, N. and Zhou, H. (2015). Oil Prices, US Stock Return, and the 
#' Dependence Between Their Quantiles. \emph{Journal of Banking & Finance}, 
#' 55, 1-12. \doi{10.1016/j.jbankfin.2015.01.013}
#'
#' @examples
#' # Generate example data
#' set.seed(42)
#' n <- 200
#' x <- rnorm(n)
#' y <- 0.5 * x + rnorm(n, sd = 0.5)
#'
#' # Run QQ regression with default quantiles
#' result <- qq_regression(y, x, verbose = FALSE)
#' 
#' # Print summary
#' print(result)
#' 
#' # View first few results
#' head(result$results)
#'
#' \donttest{
#' # Run with custom quantiles
#' result2 <- qq_regression(y, x, 
#'                          y_quantiles = seq(0.1, 0.9, by = 0.1),
#'                          x_quantiles = seq(0.1, 0.9, by = 0.1),
#'                          verbose = FALSE)
#' }
#'
#' @export
#' @importFrom quantreg rq
#' @importFrom stats quantile predict pt complete.cases var cor
qq_regression <- function(y, x, 
                          y_quantiles = seq(0.05, 0.95, by = 0.05),
                          x_quantiles = seq(0.05, 0.95, by = 0.05),
                          min_obs = 10,
                          se_method = "boot",
                          verbose = TRUE) {
  

  # Input validation
  if (!is.numeric(y) || !is.numeric(x)) {
    stop("'y' and 'x' must be numeric vectors")
  }
  
  if (length(y) != length(x)) {
    stop("'y' and 'x' must have the same length")
  }
  
  if (!all(y_quantiles > 0 & y_quantiles < 1)) {
    stop("All y_quantiles must be between 0 and 1 (exclusive)")
  }
  
  if (!all(x_quantiles > 0 & x_quantiles < 1)) {
    stop("All x_quantiles must be between 0 and 1 (exclusive)")
  }
  
  se_method <- match.arg(se_method, c("boot", "nid", "iid", "ker"))
  
  # Handle missing values
  complete_idx <- complete.cases(y, x)
  y_data <- y[complete_idx]
  x_data <- x[complete_idx]
  n_obs <- length(y_data)
  
  if (n_obs < 20) {
    stop("Insufficient observations. Need at least 20 complete cases.")
  }
  
  if (verbose) {
    message("Running Quantile-on-Quantile Regression...")
    message("Number of observations: ", n_obs)
    message("Y quantiles: ", length(y_quantiles), " levels")
    message("X quantiles: ", length(x_quantiles), " levels")
  }
  
  # Initialize results data frame
  results <- expand.grid(y_quantile = y_quantiles, x_quantile = x_quantiles)
  results$coefficient <- NA_real_
  results$std_error <- NA_real_
  results$t_value <- NA_real_
  results$p_value <- NA_real_
  results$r_squared <- NA_real_
  results$method <- NA_character_
  
  total_combinations <- nrow(results)
  
  # Define the rho (check) function for pseudo R-squared
  rho_function <- function(u, tau) {
    u * (tau - (u < 0))
  }
  
  # Main computation loop
  for (i in seq_len(total_combinations)) {
    tryCatch({
      y_tau <- results$y_quantile[i]
      x_tau <- results$x_quantile[i]
      
      # Get quantile values
      x_quant_val <- quantile(x_data, x_tau, na.rm = TRUE)
      y_quant_val <- quantile(y_data, y_tau, na.rm = TRUE)
      
      # Subset data based on x quantile
      subset_idx <- x_data <= x_quant_val
      y_subset <- y_data[subset_idx]
      x_subset <- x_data[subset_idx]
      
      if (length(y_subset) >= min_obs) {
        # Use quantile regression
        qr_model <- rq(y_subset ~ x_subset, tau = y_tau)
        qr_summary <- summary(qr_model, se = se_method)
        
        if (nrow(qr_summary$coefficients) >= 2) {
          results$coefficient[i] <- qr_summary$coefficients[2, 1]
          results$std_error[i] <- qr_summary$coefficients[2, 2]
          results$t_value[i] <- qr_summary$coefficients[2, 3]
          results$p_value[i] <- qr_summary$coefficients[2, 4]
          results$method[i] <- "quantile_regression"
          
          # Compute pseudo R-squared
          y_pred <- predict(qr_model)
          y_actual <- y_subset
          rho_sum <- sum(rho_function(y_actual - y_pred, y_tau))
          rho_null <- sum(rho_function(y_actual - quantile(y_actual, y_tau), y_tau))
          
          if (rho_null != 0) {
            results$r_squared[i] <- max(0, 1 - rho_sum / rho_null)
          }
        }
      } else {
        # Fallback to correlation-based approach
        y_binary <- as.numeric(y_data <= y_quant_val)
        x_binary <- as.numeric(x_data <= x_quant_val)
        
        if (var(x_binary) > 0 && var(y_binary) > 0) {
          correlation <- cor(y_binary, x_binary)
          n_binary <- length(x_binary)
          
          results$coefficient[i] <- correlation
          results$std_error[i] <- sqrt((1 - correlation^2) / max(1, n_binary - 2))
          results$t_value[i] <- correlation / results$std_error[i]
          results$p_value[i] <- 2 * pt(-abs(results$t_value[i]), 
                                        df = max(1, n_binary - 2))
          results$r_squared[i] <- correlation^2
          results$method[i] <- "correlation"
        }
      }
    }, error = function(e) {
      if (verbose) {
        message("Warning: Error at y_tau=", results$y_quantile[i], 
                ", x_tau=", results$x_quantile[i], ": ", e$message)
      }
    })
    
    if (verbose && i %% max(1, floor(total_combinations / 10)) == 0) {
      message("Progress: ", round(100 * i / total_combinations), "%")
    }
  }
  
  if (verbose) {
    message("QQ Regression completed successfully!")
  }
  
  # Create result object
  result <- list(
    results = results,
    y_quantiles = y_quantiles,
    x_quantiles = x_quantiles,
    n_obs = n_obs,
    call = match.call(),
    method = "Quantile-on-Quantile Regression"
  )
  
  class(result) <- "qq_regression"
  
  return(result)
}


#' @title Print Method for qq_regression Objects
#'
#' @description Prints a summary of a qq_regression object.
#'
#' @param x An object of class "qq_regression".
#' @param ... Additional arguments (currently ignored).
#'
#' @return The input object \code{x} is returned invisibly. This function 
#'   is called for its side effect of printing the QQ regression summary 
#'   to the console.
#'
#' @export
print.qq_regression <- function(x, ...) {
  cat("\n")
  cat("Quantile-on-Quantile Regression Results\n")
  cat("========================================\n\n")
  cat("Call:\n")
  print(x$call)
  cat("\n")
  cat("Number of observations:", x$n_obs, "\n")
  cat("Y quantiles:", length(x$y_quantiles), "levels\n")
  cat("X quantiles:", length(x$x_quantiles), "levels\n")
  cat("Total combinations:", nrow(x$results), "\n")
  
  # Summary statistics
  complete_results <- x$results[complete.cases(x$results), ]
  if (nrow(complete_results) > 0) {
    cat("\n")
    cat("Coefficient Summary:\n")
    cat("  Mean:", round(mean(complete_results$coefficient, na.rm = TRUE), 4), "\n")
    cat("  Min: ", round(min(complete_results$coefficient, na.rm = TRUE), 4), "\n")
    cat("  Max: ", round(max(complete_results$coefficient, na.rm = TRUE), 4), "\n")
    cat("\n")
    cat("Significant at 0.05:", 
        sum(complete_results$p_value < 0.05, na.rm = TRUE), "of", 
        nrow(complete_results), "\n")
  }
  
  invisible(x)
}


#' @title Summary Method for qq_regression Objects
#'
#' @description Provides a detailed summary of a qq_regression object.
#'
#' @param object An object of class "qq_regression".
#' @param ... Additional arguments (currently ignored).
#'
#' @return A list containing summary statistics, returned invisibly. 
#'   The function is called for its side effect of printing a detailed 
#'   summary to the console.
#'
#' @export
#' @importFrom stats median sd
summary.qq_regression <- function(object, ...) {
  complete_results <- object$results[complete.cases(object$results), ]
  
  summary_stats <- list(
    n_obs = object$n_obs,
    n_y_quantiles = length(object$y_quantiles),
    n_x_quantiles = length(object$x_quantiles),
    total_combinations = nrow(object$results),
    complete_results = nrow(complete_results),
    coef_mean = mean(complete_results$coefficient, na.rm = TRUE),
    coef_median = median(complete_results$coefficient, na.rm = TRUE),
    coef_min = min(complete_results$coefficient, na.rm = TRUE),
    coef_max = max(complete_results$coefficient, na.rm = TRUE),
    coef_sd = sd(complete_results$coefficient, na.rm = TRUE),
    rsq_mean = mean(complete_results$r_squared, na.rm = TRUE),
    rsq_median = median(complete_results$r_squared, na.rm = TRUE),
    rsq_min = min(complete_results$r_squared, na.rm = TRUE),
    rsq_max = max(complete_results$r_squared, na.rm = TRUE),
    n_significant_05 = sum(complete_results$p_value < 0.05, na.rm = TRUE),
    n_significant_01 = sum(complete_results$p_value < 0.01, na.rm = TRUE)
  )
  
  cat("\n")
  cat("Quantile-on-Quantile Regression Summary\n")
  cat("=======================================\n\n")
  
  cat("Data:\n")
  cat("  Observations:", summary_stats$n_obs, "\n")
  cat("  Y quantiles:", summary_stats$n_y_quantiles, "levels\n")
  cat("  X quantiles:", summary_stats$n_x_quantiles, "levels\n")
  cat("  Total combinations:", summary_stats$total_combinations, "\n")
  cat("  Complete results:", summary_stats$complete_results, "\n")
  
  cat("\n")
  cat("Coefficient Statistics:\n")
  cat("  Mean:  ", round(summary_stats$coef_mean, 4), "\n")
  cat("  Median:", round(summary_stats$coef_median, 4), "\n")
  cat("  Min:   ", round(summary_stats$coef_min, 4), "\n")
  cat("  Max:   ", round(summary_stats$coef_max, 4), "\n")
  cat("  SD:    ", round(summary_stats$coef_sd, 4), "\n")
  
  cat("\n")
  cat("R-squared Statistics:\n")
  cat("  Mean:  ", round(summary_stats$rsq_mean, 4), "\n")
  cat("  Median:", round(summary_stats$rsq_median, 4), "\n")
  cat("  Min:   ", round(summary_stats$rsq_min, 4), "\n")
  cat("  Max:   ", round(summary_stats$rsq_max, 4), "\n")
  
  cat("\n")
  cat("Significance:\n")
  cat("  p < 0.05:", summary_stats$n_significant_05, "of", 
      summary_stats$complete_results, "\n")
  cat("  p < 0.01:", summary_stats$n_significant_01, "of", 
      summary_stats$complete_results, "\n")
  
  invisible(summary_stats)
}


#' @title Plot Method for qq_regression Objects
#'
#' @description Creates a 3D surface plot of qq_regression results.
#'
#' @param x An object of class "qq_regression".
#' @param type Character string. Type of plot: "coefficient", "rsquared", 
#'   or "pvalue". Default is "coefficient".
#' @param colorscale Character string. Color scale to use. One of "Jet", 
#'   "BlueRed", "Viridis", or "Plasma". Default is "Jet".
#' @param ... Additional arguments passed to \code{plot_qq_3d}.
#'
#' @return A plotly object. This function is called for its side effect 
#'   of creating an interactive 3D visualization.
#'
#' @export
plot.qq_regression <- function(x, type = "coefficient", 
                                colorscale = "Jet", ...) {
  plot_qq_3d(x, type = type, colorscale = colorscale, ...)
}
