% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/calcHLGMRel.R
\name{calcHLGMRel}
\alias{calcHLGMRel}
\title{Calculate reliability using a hierarchical logistic regression model}
\usage{
calcHLGMRel(
  df = NULL,
  model = NULL,
  entity = "entity",
  y = "y",
  show.all = FALSE,
  ctrPerf = controlPerf(),
  ctrRel = controlRel()
)
}
\arguments{
\item{df}{observation-level data; if null, will use the dataframe from the model object}

\item{model}{model; if null, will use an unadjusted model}

\item{entity}{data column containing the accountable entity identifier}

\item{y}{data column containing the outcome variable}

\item{show.all}{logical parameter indicating whether all variations of reliability estimates should be calculated; default is \code{FALSE}.}

\item{ctrPerf}{parameters to control performance measure calculation}

\item{ctrRel}{parameters to control reliability estimation}
}
\value{
A list containing:
\itemize{
\item \code{fit}: fitted model
\item \code{marg.p}: marginal probability of the outcome
\item \code{entity}: list of entities
\item \code{n}: entity sample sizes
\item \code{p}: entity-level sample proportions
\item \code{p.re}: predicted entity-level outcome probabilities (i.e., shrunken estimates)
\item \code{var.b}: between-entity variance on the outcome scale
\item \code{var.w}: within-entity variance on the outcome scale
\item \code{est.HLGM.delta}: reliability estimates on the outcome scale
}

If \code{show.all} is set to \code{TRUE}, then the outputted list will also contain:
\itemize{
\item \code{var.b.HLGM.latent}: between-entity variance on the latent log-odds scale
\item \code{var.b.HLGM.delta}: between-entity variance on the outcome scale using delta method approximation
\item \code{var.b.MC}: between-entity variance on the outcome scale using Monte Carlo approximation
\item \code{var.w.latent}: within-entity variance on the latent log-odds scale
\item \code{var.w.delta}: within-entity variance on the outcome scale using delta method approximation
\item \code{var.w.MC}: within-entity variance on the outcome scale using Monte Carlo approximation
\item \code{var.w.FE}: within-entity variance calculated using fixed effect estimates of entity-level performance
\item \code{var.w.RE}: within-entity variance calculated using random effect estimates of entity-level performance
\item \code{est.HLGM.latent}: reliability estimates on latent log-odds scale
\item \code{est.HLGM.delta}: reliability estimates on outcome scale using delta approximation
\item \code{est.HLGM.MC}: reliability estimates on outcome scale using Monte Carlo approximation
\item \code{est.HLGM.FE}: reliability estimates on outcome scale using fixed effect estimates
\item \code{est.HLGM.RE}: reliability estimates on outcome scale using random effect estimates
}
}
\description{
This function estimates reliability using a hierarchical logistic regression model with random intercepts for each accountable entity.
}
\details{
Hierarchical logistic regression models are fit using \code{lme4::glmer()} with
\code{control =  lme4::glmerControl(optimizer = "bobyqa")} and \code{nAGQ = 0}.
}
\examples{
# Simulate data
df <- simulateData(n.entity = 50, n.obs = 100, mu = .2, r = .7)

# Calculate reliability
out <- calcHLGMRel(df = df, entity = 'entity', y = 'y')
summary(out$est.HLGM.delta)

# Plot reliability
plot(out$n, out$est.HLGM.delta)

## Reliability estimates from additional methods can be obtained by toggling show.all parameter
out.all <- calcHLGMRel(df = df, entity = 'entity', y = 'y', show.all = TRUE)
summary(out.all$est.HLGM.latent)
summary(out.all$est.HLGM.delta)
summary(out.all$est.HLGM.MC)
summary(out.all$est.HLGM.FE)
summary(out.all$est.HLGM.RE)

}
\references{
Goldstein H, Browne W, Rasbash J. Partitioning variation in multilevel models. Understanding statistics: statistical issues in psychology, education, and the social sciences. 2002 Dec 2;1(4):223-31.

He K, Kalbfleisch JD, Yang Y, Fei Z, Kim S, Kang J, Li Y. Inter-unit reliability for quality measure testing. Journal of hospital administration. 2019 Jan 8;8(2):1.

Hwang J, Adams JL, Paddock SM. Defining and estimating the reliability of physician quality measures in hierarchical logistic regression models. Health Services and Outcomes Research Methodology. 2021 Mar;21(1):111-30.

Nieser KJ, Harris AH. Comparing methods for assessing the reliability of health care quality measures. Statistics in Medicine. 2024 Oct 15;43(23):4575-94.
}
\author{
Kenneth Nieser (nieser@stanford.edu)
}
