#' Compute Maximum Stress-adjusted Germination (MSG)
#'
#' Calculates MSG as the maximum cumulative germination fraction across
#' all observed timepoints.
#'
#' @param germination_counts Integer vector of cumulative germination counts
#'   at each timepoint. Must be non-decreasing.
#' @param total_seeds Integer. Total number of seeds in the replicate.
#'
#' @return Numeric value in \code{[0, 1]} representing the maximum germination
#'   fraction.
#'
#' @details
#' MSG captures the overall germination capacity under stress:
#' \deqn{MSG = \max(g_1, g_2, \ldots, g_k) / N}
#' where \eqn{g_t} is the cumulative germination count at timepoint \eqn{t}
#' and \eqn{N} is the total seed count.
#'
#' @examples
#' # 25 seeds, counts at days 3, 5, 7
#' compute_msg(c(5, 15, 20), total_seeds = 25)
#' # Returns 0.8
#'
#' # Zero germination
#' compute_msg(c(0, 0, 0), total_seeds = 25)
#' # Returns 0
#'
#' @export
compute_msg <- function(germination_counts, total_seeds) {
  if (!is.numeric(germination_counts) || length(germination_counts) == 0) {
    stop("germination_counts must be a non-empty numeric vector.")
  }
  if (!is.numeric(total_seeds) || length(total_seeds) != 1 || total_seeds <= 0) {
    stop("total_seeds must be a single positive number.")
  }
  if (any(germination_counts < 0)) {
    stop("germination_counts must be non-negative.")
  }
  if (max(germination_counts) > total_seeds) {
    stop("germination_counts cannot exceed total_seeds.")
  }

  max(germination_counts) / total_seeds
}


#' Compute Maximum Rate of Germination (MRG)
#'
#' Calculates MRG as the scaled average germination rate across consecutive
#' observation intervals.
#'
#' @param germination_counts Integer vector of cumulative germination counts
#'   at each timepoint.
#' @param timepoints Numeric vector of observation times (e.g., days). Must
#'   be the same length as \code{germination_counts}.
#' @param total_seeds Integer. Total number of seeds in the replicate.
#'
#' @return Numeric value >= 0 representing the scaled germination rate.
#'
#' @details
#' MRG measures how rapidly germination proceeds. For each interval between
#' consecutive timepoints, the germination rate is computed as the change in
#' germination count divided by the interval length and total seeds. The
#' average rate is then scaled by the number of timepoints:
#' \deqn{MRG = \frac{1}{k-1} \sum_{t=2}^{k} \frac{\max(0, g_t - g_{t-1})}{(t_t - t_{t-1}) \cdot N} \cdot k}
#'
#' For a single timepoint, MRG defaults to 0.1 (rate not computable).
#'
#' @examples
#' # Standard 3-timepoint series
#' compute_mrg(c(5, 15, 20), timepoints = c(3, 5, 7), total_seeds = 25)
#'
#' # Single timepoint
#' compute_mrg(c(10), timepoints = c(3), total_seeds = 25)
#' # Returns 0.1
#'
#' @export
compute_mrg <- function(germination_counts, timepoints, total_seeds) {
  if (!is.numeric(germination_counts) || length(germination_counts) == 0) {
    stop("germination_counts must be a non-empty numeric vector.")
  }
  if (!is.numeric(timepoints) || length(timepoints) == 0) {
    stop("timepoints must be a non-empty numeric vector.")
  }
  if (length(germination_counts) != length(timepoints)) {
    stop("germination_counts and timepoints must have the same length.")
  }
  if (!is.numeric(total_seeds) || length(total_seeds) != 1 || total_seeds <= 0) {
    stop("total_seeds must be a single positive number.")
  }

  k <- length(germination_counts)

  # Single timepoint: rate not computable

  if (k == 1) {
    return(0.1)
  }

  # Compute interval rates
  rates <- numeric(k - 1)
  for (i in seq_len(k - 1)) {
    delta_g <- max(0, germination_counts[i + 1] - germination_counts[i])
    delta_t <- timepoints[i + 1] - timepoints[i]
    if (delta_t <= 0) {
      stop("timepoints must be strictly increasing.")
    }
    rates[i] <- delta_g / (delta_t * total_seeds)
  }

  # Average rate scaled by number of timepoints
  mean(rates) * k
}


#' Compute Complementary Mean Time to Germination (cMTG)
#'
#' Calculates cMTG as the complement of the normalized time to 50\%
#' germination, bounded in \code{[0, 1]}.
#'
#' @param germination_counts Integer vector of cumulative germination counts
#'   at each timepoint.
#' @param timepoints Numeric vector of observation times.
#' @param total_seeds Integer. Total number of seeds in the replicate.
#'
#' @return Numeric value in \code{[0, 1]}. Higher values indicate faster
#'   germination. Returns \code{1 - (first_timepoint / max_timepoint)}
#'   as default when 50\% threshold is not crossed.
#'
#' @details
#' cMTG inverts the time-to-germination metric so that higher values
#' indicate better performance (faster germination under stress):
#' \deqn{cMTG = 1 - \frac{t_{50}}{t_{max}}}
#' where \eqn{t_{50}} is the interpolated time at which 50\% of the final
#' germination count is reached, and \eqn{t_{max}} is the maximum
#' observation time. If 50\% germination is never reached, the default
#' normalization uses the first timepoint.
#'
#' @examples
#' # Rapid germination: most seeds by day 3
#' compute_cmtg(c(20, 23, 25), timepoints = c(3, 5, 7), total_seeds = 25)
#'
#' # Slow germination: 50% not reached until late
#' compute_cmtg(c(2, 8, 20), timepoints = c(3, 5, 7), total_seeds = 25)
#'
#' # No germination
#' compute_cmtg(c(0, 0, 0), timepoints = c(3, 5, 7), total_seeds = 25)
#'
#' @export
compute_cmtg <- function(germination_counts, timepoints, total_seeds) {
  if (!is.numeric(germination_counts) || length(germination_counts) == 0) {
    stop("germination_counts must be a non-empty numeric vector.")
  }
  if (!is.numeric(timepoints) || length(timepoints) == 0) {
    stop("timepoints must be a non-empty numeric vector.")
  }
  if (length(germination_counts) != length(timepoints)) {
    stop("germination_counts and timepoints must have the same length.")
  }

  k <- length(germination_counts)
  t_max <- max(timepoints)
  final_count <- max(germination_counts)

  # No germination or single timepoint: use default normalization
  if (final_count == 0 || k == 1) {
    return(1 - (timepoints[1] / t_max))
  }

  # Find 50% threshold
  threshold <- final_count * 0.5

  # Check if threshold already exceeded at first timepoint
  if (germination_counts[1] >= threshold) {
    return(1 - (timepoints[1] / t_max))
  }

  # Interpolate time to 50%
  for (i in seq_len(k - 1)) {
    if (germination_counts[i] < threshold && germination_counts[i + 1] >= threshold) {
      # Linear interpolation within this interval
      frac <- (threshold - germination_counts[i]) /
              (germination_counts[i + 1] - germination_counts[i])
      t50 <- timepoints[i] + frac * (timepoints[i + 1] - timepoints[i])
      return(1 - (t50 / t_max))
    }
  }

  # Threshold never crossed (e.g., all counts below 50%)
  1 - (timepoints[1] / t_max)
}


#' Compute Radicle Vigor Score (RVS)
#'
#' Calculates a continuous radicle vigor score on a \code{[0, 1]} scale
#' from radicle emergence counts and total seed counts.
#'
#' @param radicle_count Integer. Number of seeds with visible radicle
#'   emergence at the final observation.
#' @param total_seeds Integer. Total number of seeds in the replicate.
#'
#' @return Numeric value in \code{[0, 1]} representing the fraction of
#'   seeds exhibiting radicle vigor.
#'
#' @details
#' RVS provides a continuous measure of early seedling vigor,
#' complementing the germination count metrics (MSG, MRG, cMTG).
#' Unlike the discrete Radicle Vigor Factor (RVF = 1.0, 1.05, or 1.10)
#' used in the geometric PSRI (\code{PSRICalc}), RVS varies continuously:
#' \deqn{RVS = \frac{\text{radicle\_count}}{\text{total\_seeds}}}
#'
#' This continuous formulation integrates naturally with the softmax
#' weighting scheme, where RVS is treated as a fourth component alongside
#' MSG, MRG, and cMTG.
#'
#' When \code{radicle_count} is \code{NULL} or \code{NA}, the function
#' returns 0, effectively excluding radicle information from the PSRI.
#'
#' @examples
#' compute_rvs(18, total_seeds = 25)
#' # Returns 0.72
#'
#' compute_rvs(0, total_seeds = 25)
#' # Returns 0
#'
#' # No radicle data available
#' compute_rvs(NULL, total_seeds = 25)
#' # Returns 0
#'
#' @export
compute_rvs <- function(radicle_count, total_seeds) {
  if (is.null(radicle_count) || length(radicle_count) == 0 ||
      is.na(radicle_count)) {
    return(0)
  }
  if (!is.numeric(radicle_count) || length(radicle_count) != 1 ||
      radicle_count < 0) {
    stop("radicle_count must be a single non-negative number or NULL/NA.")
  }
  if (!is.numeric(total_seeds) || length(total_seeds) != 1 || total_seeds <= 0) {
    stop("total_seeds must be a single positive number.")
  }
  if (radicle_count > total_seeds) {
    stop("radicle_count cannot exceed total_seeds.")
  }

  radicle_count / total_seeds
}
