
# PERSUADE

**ParamEtRic SUrvivAl moDel sElection**

A standardized R tool to support the selection, comparison, and
communication of parametric survival models, and their implementation in
decision-analytic models.

------------------------------------------------------------------------

## Overview

PERSUADE guides analysts through a reproducible workflow for survival
analysis and model selection. It provides:

- Kaplan-Meier visualizations and diagnostics
- Assessment of proportional hazards (PH) assumption
- Observed and predicted hazard function plots
- Fit comparisons of standard, spline-based, and cure models
- Goodness-of-fit statistics (AIC/BIC)
- Explore model extrapolation beyond observed data
- Automated PDF reports for communication
- Export the parametric survival model parameters for use in
  decision-analytic models

The framework is designed for transparency and consistency when survival
analysis feeds into decision-analytic models.

------------------------------------------------------------------------

## Prerequisites

- R (version compatible with survival and other dependencies)
- Recommended: RStudio for better environment handling
- Install PERSUADE package using `install.packages("PERSUADE")`
- Alternatively: to install the development version of PERSUADE:
  `devtools::install_github("Bram-R/PERSUADE", quiet = TRUE, upgrade = "never")`
- Load PERSUADE package: `library(PERSUADE)`

## Quick start

The easiest way to get started is by running the example workflow
script: `PERSUADE_example_workflow.R` using:
`file.edit(system.file("example_workflow", "PERSUADE_example_workflow.R", package = "PERSUADE"))`

This script:

- Loads example survival data.
- Runs the main PERSUADE analysis (`f_PERSUADE()`).
- Summarizes results with `print()`, `summary()`, and `plot()`.
- Generates a ready-to-share PDF report via `f_generate_report()`.
- Exports fitted model parameters to .csv.

------------------------------------------------------------------------

## Step-by-step workflow

### 1. Load PERSUADE package

``` r
library(PERSUADE)
```

### 2. Run example analysis

``` r
PERSUADE <- f_PERSUADE(
  name = "BC_OS",
  years = flexsurv::bc$recyrs,   # time-to-event
  status = flexsurv::bc$censrec, # event indicator
  group = flexsurv::bc$group,    # grouping variable
  strata = TRUE,
  spline_mod = TRUE,
  cure_mod = TRUE,
  time_unit = 1/12,              # months in years
  time_horizon = 40,
  time_pred_surv_table = c(0, 1, 2, 5, 10, 20, 30, 40)
)
```

### 3. Inspect the object

``` r
print(PERSUADE)
```

### 4. Summarize results

``` r
# Kaplan-Meier estimates
summary(PERSUADE, type = "km")

# Survival probabilities at time points
summary(PERSUADE, type = "surv_probs")

# Goodness-of-fit for standard parametric models
summary(PERSUADE, type = "gof")

# Goodness-of-fit for spline-based models
summary(PERSUADE, type = "gof_spline")

# Goodness-of-fit for cure models
summary(PERSUADE, type = "gof_cure")
```

### 5. Visualize results

``` r
# Set colour palette for Figures
palette(rainbow(n = 9, s = 1, v = 1, start = 0, end = max(1, 9 - 1)/9, alpha = 1)) 

# Kaplan-Meier curves
plot(PERSUADE, type = "km")

# Proportional hazards diagnostics
plot(PERSUADE, type = "ph")

# Observed smoothed hazard
plot(PERSUADE, type = "hr")

# Standard parametric models
plot(PERSUADE, type = "param_models")

# Spline-based models
plot(PERSUADE, type = "spline_models")

# Cure models
plot(PERSUADE, type = "cure_models")

# Set colour palette to default
palette("default") 
```

### 6. Generate the report

``` r
f_generate_report(PERSUADE)
```

### 7. Export model parameters

``` r
# Copy to clipboard (for copying in PERSUADE Excel template.xltx)
write.table(PERSUADE$surv_model_excel, "clipboard-128", sep = "\t", col.names = FALSE)

# Export to CSV (change `tempdir()` into `getwd()` for copying to working directory)
write.csv(PERSUADE$surv_model_excel, 
          file.path(tempdir(), paste0(name, "_output"),
                    "PERSUADE_Time-to-event_models_parameters_comma.csv"))
write.csv2(PERSUADE$surv_model_excel, 
           file.path(tempdir(), paste0(name, "_output"),
                     "PERSUADE_Time-to-event_models_parameters_semicolon.csv"))
```

### 8. Obtain Excel template to incorporate model parameters in decision-analytic model

``` r
f_get_excel_template()
```

------------------------------------------------------------------------

## Workflow diagram

Visualize the process via the included diagram:  
![PERSUADE Workflow Overview](PERSUADE_figure_process.png)

*(Alternatively, open the `.vsdx` version if preferred.)*

------------------------------------------------------------------------

### Tips for running your own analyses

Start by running the example script to familiarize yourself with the
workflow. For your own data:

- copy `PERSUADE_example_workflow.R` using:
  `file.edit(system.file("example_workflow", "PERSUADE_example_workflow.R", package = "PERSUADE"))`;
  see also vignette using: `browseVignettes("PERSUADE")`
- replace the input section with your dataset
- set the project name and outcome identifier using the `name`
  parameter  
- within the call to `f_PERSUADE(...)`, toggle options (`TRUE`/`FALSE`)
  for the analyses you want to run

------------------------------------------------------------------------

## Extending or customizing

- Add new model types: incorporate additional parametric distributions
  or non-parametric methods by extending `f_PERSUADE()`.  
- Refine plots: customize visualizations in
  `PERSUADE_output_functions.R` or adopt themes for publication-quality
  graphics.  
- Automate Reporting: enhance the ‘rmarkdown’ template for
  production-ready reports.

------------------------------------------------------------------------

## Acknowledgments & history

- Origins: this repository represents an ongoing effort to standardize
  survival model workflows in decision analysis.  
- Contributors: developed by **Bram-R**
  (<https://orcid.org/0000-0001-5785-9228>) and **Xa4P**
  (<https://orcid.org/0000-0003-3563-0013>); contributions are welcome
  via pull requests or issues.

------------------------------------------------------------------------

## License & contact

Licensed under
[**GPL3**](https://github.com/Bram-R/PERSUADE/tree/master?tab=GPL-3.0-1-ov-file#readme).  
For questions, enhancements, or collaboration, reach out via GitHub
discussions or issues.
