
#' @title Percent Disease Index (PDI)
#'
#' @description
#' Calculates Percent Disease Index (PDI) based on disease grades
#' and their corresponding frequencies.
#'
#' @param freq A numeric vector of frequencies for each disease grade
#' (starting from grade 0).
#'
#' @return A numeric value representing Percent Disease Index (PDI).
#'
#' @details
#' PDI is calculated as:
#' \deqn{PDI = ( \sum (grade \times frequency) / (N \times max\_grade) ) \times 100}
#'
#' where N is the total number of observations.
#'
#' @export
#'
#' @examples {
#' freq <- c(10, 8, 6, 4, 2, 1)
#' PDI(freq)
#' }
PDI <- function(freq) {

  # Check input
  if (!is.numeric(freq) || any(freq < 0)) {
    stop("freq must be a numeric vector of non-negative values")
  }

  n_grades <- length(freq)
  grades <- 0:(n_grades - 1)

  numerator <- sum(grades * freq)
  total_parts <- sum(freq)
  max_score <- max(grades)

  pdi <- (numerator / (total_parts * max_score)) * 100
  return(round(pdi, 2))
}
