#' Calculate Total N derived from Fertilizer (TNdfF)
#' @description
#' The total quantity of plant or soil N that is derived from fertilizer.
#' Formula: TNdfF = (NdfF/100) * (PlantN or SoilN)
#' @param NdfF N derived from Fertilizer expressed as a percentage (0-100).
#' @param PlantN Optional numeric vector for plant N content.
#' @param SoilN Optional numeric vector for soil N content.
#' @return A numeric vector of TNdfF values.
#' @examples
#' NdfF <- c(15, 20)
#' PlantN <- c(3.0, 3.5)
#' 
#' # Case 1: Using Plant N
#' TNdfF(NdfF, PlantN = PlantN)
#' 
#' # Case 2: Using Soil N (must specify SoilN explicitly)
#' SoilN <- c(100, 120)
#' TNdfF(NdfF, SoilN = SoilN)
#' @export
TNdfF <- function(NdfF, PlantN = NULL, SoilN = NULL) {
  if (!is.null(PlantN)) {
    return((NdfF / 100) * PlantN)
  } else if (!is.null(SoilN)) {
    return((NdfF / 100) * SoilN)
  } else {
    stop("Please provide either PlantN or SoilN.")
  }
}