\name{l1ccc}
\alias{l1ccc}
\title{L1 concordance correlation coefficient}
\description{
  Calculates \emph{L1} concordance correlation coefficient for evaluating the degree 
  of agreement between measurements generated by two different methods.
}
\usage{
l1ccc(x, data, equal.means = FALSE, boots = TRUE, nsamples = 1000, subset, na.action)
}
\arguments{
  \item{x}{ a formula or a numeric matrix or an object that can be coerced to a numeric matrix.}
  \item{data}{ an optional data frame (or similar: see \code{\link{model.frame}}), used only if
    \code{x} is a formula. By default the variables are taken from \code{environment(formula)}.}
  \item{equal.means}{ logical, should the means of the measuring devices be considered equal? 
    In which case the restricted estimation is carried out under this assumption.}
  \item{boots}{ logical, hould use bootstrap to approximate the variances of the \emph{L1} estimators.}
  \item{nsamples}{ number of bootstrap samples (default to 1000), only used if \code{boots = TRUE}.}
  \item{subset}{ an optional expression indicating the subset of the rows of data that should 
    be used in the fitting process.}
  \item{na.action}{ a function that indicates what should happen when the data contain NAs.}
}
\value{
  A list with class \code{'l1ccc'} containing the following named components:
  \item{call}{ a list containing an image of the \code{ccc} call that produced the object.}
  \item{x}{ \code{data.frame} used in the estimation process.}
  \item{rho1}{ \emph{L1} estimate of the concordance correlation coefficient.}
  \item{var.rho1}{ approximate variance of the \emph{L1} concordance correlation coefficient, 
    only present if \code{boots = TRUE}.}
  \item{L1}{ list with \emph{L1} estimates using Laplace, and normal distributions, and \emph{U}-statistics.}
  \item{Lin}{ Lin's concordance correlation coefficient under the Laplace distribution.}
  \item{ustat}{ \emph{L1} estimation of concordance coefficient using \emph{U}-statistics, 
    a list containing the following elements \code{rho1}, \code{var.rho1}, \code{ustat}, 
    and \code{cov}.}
  \item{center}{ the estimated mean vector.}
  \item{Scatter}{ the estimated Scatter (or Scale) matrix.}
  \item{logLik}{ the log-likelihood at convergence.}
  \item{weights}{ estimated weights corresponding to the Laplace distribution.}
  \item{Restricted}{ available only if \code{equal.means = TRUE}, in which case this element 
    corresponds to a list containing the following elements \code{ccc}, \code{rho1}, \code{var.rho1}, 
    \code{accuracy}, \code{precision}, \code{shifts}, and information about the restrcited \code{Fitted}
    model.}
}
\references{
  King, T.S., Chinchilli, V.M. (2001).
  A generalized concordance correlation coefficient for continuous and categorical data.
  \emph{Statistics in Medicine} \bold{20}, 2131-2147.

  King, T.S., Chinchilli, V.M. (2001).
  Robust estimators of the concordance correlation coefficient.
  \emph{Journal of Biopharmaceutical Statistics} \bold{11}, 83-105.

  Lin, L. (1989). 
  A concordance correlation coefficient to evaluate reproducibility. 
  \emph{Biometrics} \bold{45}, 255-268.

  Vallejos, R., Osorio, F., Ferrer, C. (2025).
  A new coefficient to measure agreement between two continuous variables.
  doi: 10.48550/arXiv.2507.07913.
}
\examples{
## data from Bland and Altman (1986). The Lancet 327, 307-310.
x <- list(Large = c(494,395,516,434,476,557,413,442,650,433,
          417,656,267,478,178,423,427),
          Mini  = c(512,430,520,428,500,600,364,380,658,445,
          432,626,260,477,259,350,451))
x <- as.data.frame(x)

plot(Mini ~ Large, data = x, xlim = c(100,800), ylim = c(100,800),
     xlab = "PERF by Large meter", ylab = "PERF by Mini meter")
abline(c(0,1), col = "gray", lwd = 2)

## estimating L1 concordance coefficient
z <- l1ccc(~ Mini + Large, data = x, boots = FALSE)
z
## output:
# Call:
# l1ccc(x = ~ Mini + Large, data = x, boots = FALSE)
#
# L1 coefficients using:
# Laplace  Gaussian  U-statistic 
#  0.7456    0.7607       0.7642 
#
# Lin's coefficients:
# estimate  accuracy precision 
#   0.9395    0.9974    0.9419 
}

\keyword{univar}
