% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/BayesPET.R
\name{fit_models}
\alias{fit_models}
\title{Fit enrollment, event-time, and censoring models to clinical
trial data and return posterior draws model parameters}
\usage{
fit_models(
  data.enroll,
  data.eventcensor,
  blinded = TRUE,
  p_trt = NULL,
  hyperparams_enroll = list(),
  hyperparams_event = list(),
  hyperparams_censor = list(),
  chains = 4,
  iter = 4000,
  mc.cores = 1,
  warmup = floor(iter/2),
  seed = list(123),
  refresh = 0,
  control = list(list(adapt_delta = 0.95)),
  return_fit = FALSE,
  quiet = TRUE
)
}
\arguments{
\item{data.enroll}{A data frame of enrollment information up to the analysis time.
Must contain the columns:
\itemize{
\item \code{interarrivaltime}: Numeric vector of interarrival times (> 0).
\item \code{enrollstatus}: Integer vector coded \code{1} = enrolled,
\code{0} = administratively censored.
}
Any additional columns are ignored.}

\item{data.eventcensor}{A data frame of observed event/censoring outcomes at the analysis time.
Must contain (at minimum) the following columns:
\itemize{
\item \code{time}: observed follow-up time (event or censoring time);
must be numeric and \eqn{> 0}.
\item \code{eventstatus}: event indicators (\code{1} = event, \code{0} = right-censored).
\item \code{censorstatus}: random censoring indicators \eqn{\delta_{ci}}
(\code{1} = random censoring observed,
\code{0} = no random censoring, including administrative censoring
or event observed).
}

If \code{blinded = FALSE}, \code{data.eventcensor} must also contain:
\itemize{
\item \code{trt}: observed treatment assignment indicators coded \code{0/1}.
}

The column \code{No} (representing a subject index) may be included but is not required.
Any additional columns (other than \code{No}, \code{trt}, \code{time}, \code{eventstatus},
and \code{censorstatus}) are treated as numeric baseline covariates and will be used if present.}

\item{blinded}{Logical. If \code{TRUE} (default), the interim analysis is blinded and treatment assignments
for current subjects are not observed in the data. If \code{FALSE}, the analysis
is unblinded and observed treatment assignments are used.}

\item{p_trt}{Numeric scalar in \eqn{[0,1]} giving the prespecified randomization probability
of assignment to the experimental treatment arm. Required only if \code{blinded = TRUE};
ignored otherwise. Defaults to \code{NULL}.}

\item{hyperparams_enroll}{A named list of prior hyperparameters for the enrollment model.
If it is empty (\code{list()}), default values
are assigned internally.
\itemize{
\item \code{alpha_mu}: Shape parameter of the Gamma prior for the
enrollment rate \eqn{\mu} (default: \eqn{0.1}).
\item \code{beta_mu}: Rate parameter of the Gamma prior for the
enrollment rate \eqn{\mu} (default: \eqn{0.1}).
}
No other hyperparameters are allowed.}

\item{hyperparams_event}{A named list of prior hyperparameters for the event-time model.
If it is empty (\code{list()}), default values
are assigned internally.
\itemize{
\item \code{alpha_e}, \code{beta_e}: Shape and rate parameters of the Gamma prior for
the baseline scale parameter \eqn{\lambda_{0e}}
(defaults: \eqn{0.1}, \eqn{0.1}).
\item \code{alpha_re}, \code{beta_re}: Shape and rate parameters of the Gamma prior for
the baseline shape parameter \eqn{\rho_e}
(defaults: \eqn{0.1}, \eqn{0.1}).
\item \code{mu_eta}, \code{sigma_eta}: Mean and standard deviation of the
Normal prior for the treatment effect (log hazard ratio) \eqn{\eta}
(defaults: \eqn{0}, \eqn{\sqrt{2}}).
\item \code{mu_be}, \code{sigma_be}: Mean and standard deviation of the Normal
prior for the covariate effects \eqn{\beta_{ej}}
(defaults: \eqn{0}, \eqn{\sqrt{10}}).
}
When \code{blinded = TRUE}, the prior for the treatment effect \eqn{\eta}
is a truncated Normal distribution on \eqn{[0, \infty)} with parameters
\code{mu_eta} and \code{sigma_eta}. When \code{blinded = FALSE}, the prior
for \eqn{\eta} is an untruncated Normal distribution.
No other hyperparameters are allowed.}

\item{hyperparams_censor}{A named list of prior hyperparameters for the censoring model.
If it is empty (\code{list()}), default values
are assigned internally.
\itemize{
\item \code{alpha_c, beta_c}: Shape and rate parameters of the Gamma prior for
the baseline scale parameter \eqn{\lambda_{0c}}
(defaults: \eqn{0.1}, \eqn{0.1}).
\item \code{alpha_rc, beta_rc}: Shape and rate parameters of the Gamma prior for
the baseline shape parameter \eqn{\rho_c}
(defaults: \eqn{0.1}, \eqn{0.1}).
\item \code{mu_bc, sigma_bc}: Mean and standard deviation of the Normal
prior for the covariate effects \eqn{\beta_{cj}}
(defaults: \eqn{0}, \eqn{\sqrt{10}}).
}
No other hyperparameters are allowed.}

\item{chains}{Number of Markov chain Monte Carlo (MCMC) chains. Defaults to \code{4}.}

\item{iter}{Number of iterations per chain (including warmup). Defaults to \code{4000}.}

\item{mc.cores}{Integer. Number of CPU cores to use when executing Markov
chains in parallel via \code{\link[rstan]{sampling}}. Defaults to \code{1}.
We recommend setting \code{mc.cores} to the maximum number of processors
supported by the available hardware and memory, up to the number of
chains.}

\item{warmup}{Number of warmup (burn-in) iterations per chain. Must be strictly smaller
than \code{iter}. Defaults to \code{floor(iter / 2)}.}

\item{seed}{Optional random seed(s) passed to \code{\link[rstan]{sampling}} for reproducibility.
Can be specified as:
\itemize{
\item a single integer or \code{NULL}, in which case the same seed
is used for all three submodels, or
\item a list of up to three integers or \code{NULL}s, recycled to length 3,
corresponding to the enrollment, censoring, and event-time models,
respectively.
}
Use \code{NULL} to allow 'Stan' to select a seed internally. Defaults to \code{list(123)}.}

\item{refresh}{Frequency of progress updates from \code{\link[rstan]{sampling}}. Set to \code{0} (default) to suppress output.}

\item{control}{Sampler control settings passed to \code{\link[rstan]{sampling}} for the three
submodels (enrollment, censoring, and event-time).
Can be specified as:
\itemize{
\item a single named list of control parameters (shared across all
three submodels), or
\item a list of up to three named lists, recycled to length 3, giving
separate control settings for the enrollment, censoring, and
event-time models, respectively.
}
Typical entries include \code{adapt_delta} and \code{max_treedepth}.

Defaults to \code{list(list(adapt_delta = 0.95))}.}

\item{return_fit}{Logical; if \code{TRUE}, also return the underlying 'rstan' \code{stanfit} objects for the
enrollment, censoring, and event models. Defaults to \code{FALSE}.}

\item{quiet}{Logical. If \code{TRUE} (default), suppress messages and diagnostic warnings
from 'Stan' during model fitting. Useful for large simulation studies.}
}
\value{
An object of class \code{"BayesPET_fit"}, a named list containing
posterior draws and related information from the fitted models, with elements:
\itemize{
\item \code{blinded}: Logical; indicates whether the analysis is blinded,

\item \code{mu}:
Posterior draws of the enrollment rate \eqn{\mu}.

\item \code{rho_c}:
Posterior draws of the censoring-model Weibull shape parameter \eqn{\rho_c}.

\item \code{lambda_0c}:
Posterior draws of the censoring-model baseline Weibull scale parameter \eqn{\lambda_{0c}}.

\item \code{beta_c}:
Posterior draws of the censoring-model covariate log hazard ratios
\eqn{\boldsymbol{\beta}_c}, or \code{NULL} if no covariates are included.

\item \code{eta}:
Posterior draws of the treatment log hazard ratio \eqn{\eta}.

\item \code{rho_e}:
Posterior draws of the event-model Weibull shape parameter \eqn{\rho_e}.

\item \code{lambda_0e}:
Posterior draws of the event-model baseline Weibull scale parameter \eqn{\lambda_{0e}}.

\item \code{beta_e}:
Posterior draws of the event-model covariate log hazard ratios
\eqn{\boldsymbol{\beta}_e}, or \code{NULL} if no covariates are included.

\item \code{treatment_ind}:
Observed treatment assignments \eqn{x_i \in \{0,1\}} (only returned if
\code{blinded = FALSE}).

\item \code{x}:
Posterior draws of latent treatment assignments \eqn{x_i \in \{0,1\}}
(only returned if \code{blinded = TRUE}).

\item \code{fit}:
A list with components \code{enroll}, \code{censor}, and \code{event}
containing the underlying 'rstan' \code{stanfit} objects (present only if \code{return_fit = TRUE}).
}
The default \code{\link{print}} method displays a concise overview.
}
\description{
Fits the enrollment, event-time, and censoring models to trial data
and returns posterior draws of model parameters.
}
\details{
This function fits three submodels: an enrollment model,
a censoring model, and an event-time model conditional on the given trial data.
If treatment assignments are known, the event-time model is fit using
\code{\link{fit_event_unblind}}; otherwise, a blinded event-time model
is fit using \cr \code{\link{fit_event_blind}}.
Technical details of the likelihoods, priors and parameterizations are documented in
\code{\link{fit_enroll}}, \code{\link{fit_censor}},
\code{\link{fit_event_unblind}}, and \code{\link{fit_event_blind}}.
}
\examples{
\dontshow{if (requireNamespace("rstan", quietly = TRUE)) withAutoprint(\{ # examplesIf}
\donttest{
data(data_example)
example_enroll <- data_example$example_enroll
example_eventcensor <- data_example$example_eventcensor

# Unblinded analysis
## Use 2 chains and iter = 2000 here to reduce runtime for the example;
## use more chains in real analyses.
fit.unblind <- fit_models(
  data.enroll = example_enroll,
  data.eventcensor = example_eventcensor,
  blinded = FALSE,
  chains = 2, iter = 2000, seed = list(123),
  return_fit = TRUE, mc.cores = 1, quiet = FALSE
)

# Blinded analysis
example_eventcensor.blind <- example_eventcensor
example_eventcensor.blind$trt <- NA
## Use 2 chains and iter = 2000 here to reduce runtime for the example;
## use more chains in real analyses.
fit.blind <- fit_models(
  data.enroll = example_enroll,
  data.eventcensor = example_eventcensor.blind,
  blinded = TRUE, p_trt = 0.5,
  chains = 2, iter = 2000, seed = list(123),
  return_fit = TRUE, mc.cores = 1, quiet = FALSE
)

print(fit.unblind)
summary(fit.unblind$eta)

print(fit.blind)
summary(fit.blind$eta)
}
\dontshow{\}) # examplesIf}
}
\seealso{
Other BayesPET model fitting: \code{\link{fit_censor}}, \code{\link{fit_enroll}},
\code{\link{fit_event_blind}}, \code{\link{fit_event_unblind}}, \cr \code{\link{print.BayesPET_fit}}
}
