% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/num-rpiq.R
\name{rpiq}
\alias{rpiq}
\alias{rpiq.data.frame}
\alias{rpiq_vec}
\title{Ratio of performance to inter-quartile}
\usage{
rpiq(data, ...)

\method{rpiq}{data.frame}(data, truth, estimate, na_rm = TRUE, case_weights = NULL, ...)

rpiq_vec(truth, estimate, na_rm = TRUE, case_weights = NULL, ...)
}
\arguments{
\item{data}{A \code{data.frame} containing the columns specified by the \code{truth}
and \code{estimate} arguments.}

\item{...}{Not currently used.}

\item{truth}{The column identifier for the true results
(that is \code{numeric}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:topic-inject]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{numeric} vector.}

\item{estimate}{The column identifier for the predicted
results (that is also \code{numeric}). As with \code{truth} this can be
specified different ways but the primary method is to use an
unquoted variable name. For \verb{_vec()} functions, a \code{numeric} vector.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}

\item{case_weights}{The optional column identifier for case weights. This
should be an unquoted column name that evaluates to a numeric column in
\code{data}. For \verb{_vec()} functions, a numeric vector,
\code{\link[hardhat:importance_weights]{hardhat::importance_weights()}}, or \code{\link[hardhat:frequency_weights]{hardhat::frequency_weights()}}.}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{rpd_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
These functions are appropriate for cases where the model outcome is a
numeric. The ratio of performance to deviation
(\code{\link[=rpd]{rpd()}}) and the ratio of performance to inter-quartile (\code{\link[=rpiq]{rpiq()}})
are both measures of consistency/correlation between observed
and predicted values (and not of accuracy).
}
\details{
In the field of spectroscopy in particular, the ratio
of performance to deviation (RPD) has been used as the standard
way to report the quality of a model. It is the ratio between
the standard deviation of a variable and the standard error of
prediction of that variable by a given model. However, its
systematic use has been criticized by several authors, since
using the standard deviation to represent the spread of a
variable can be misleading on skewed dataset. The ratio of
performance to inter-quartile has been introduced by
Bellon-Maurel et al. (2010) to address some of these issues, and
generalise the RPD to non-normally distributed variables.
}
\examples{
# Supply truth and predictions as bare column names
rpd(solubility_test, solubility, prediction)

library(dplyr)

set.seed(1234)
size <- 100
times <- 10

# create 10 resamples
solubility_resampled <- bind_rows(
  replicate(
    n = times,
    expr = sample_n(solubility_test, size, replace = TRUE),
    simplify = FALSE
  ),
  .id = "resample"
)

# Compute the metric by group
metric_results <- solubility_resampled \%>\%
  group_by(resample) \%>\%
  rpd(solubility, prediction)

metric_results

# Resampled mean estimate
metric_results \%>\%
  summarise(avg_estimate = mean(.estimate))
}
\references{
Williams, P.C. (1987) Variables affecting near-infrared
reflectance spectroscopic analysis. In: Near Infrared Technology
in the Agriculture and Food Industries. 1st Ed. P.Williams and
K.Norris, Eds. Am. Cereal Assoc. Cereal Chem., St. Paul, MN.

Bellon-Maurel, V., Fernandez-Ahumada, E., Palagos, B., Roger,
J.M. and McBratney, A., (2010). Critical review of chemometric
indicators commonly used for assessing the quality of the
prediction of soil attributes by NIR spectroscopy. TrAC Trends
in Analytical Chemistry, 29(9), pp.1073-1081.
}
\seealso{
The closely related deviation metric: \code{\link[=rpd]{rpd()}}

Other numeric metrics: 
\code{\link{ccc}()},
\code{\link{huber_loss_pseudo}()},
\code{\link{huber_loss}()},
\code{\link{iic}()},
\code{\link{mae}()},
\code{\link{mape}()},
\code{\link{mase}()},
\code{\link{mpe}()},
\code{\link{msd}()},
\code{\link{poisson_log_loss}()},
\code{\link{rmse}()},
\code{\link{rpd}()},
\code{\link{rsq_trad}()},
\code{\link{rsq}()},
\code{\link{smape}()}

Other consistency metrics: 
\code{\link{ccc}()},
\code{\link{rpd}()},
\code{\link{rsq_trad}()},
\code{\link{rsq}()}
}
\author{
Pierre Roudier
}
\concept{consistency metrics}
\concept{numeric metrics}
