% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/class-precision.R
\name{precision}
\alias{precision}
\alias{precision.data.frame}
\alias{precision_vec}
\title{Precision}
\usage{
precision(data, ...)

\method{precision}{data.frame}(data, truth, estimate, estimator = NULL, na_rm = TRUE, ...)

precision_vec(truth, estimate, estimator = NULL, na_rm = TRUE, ...)
}
\arguments{
\item{data}{Either a \code{data.frame} containing the \code{truth} and \code{estimate}
columns, or a \code{table}/\code{matrix} where the true class results should be
in the columns of the table.}

\item{...}{Not currently used.}

\item{truth}{The column identifier for the true class results
(that is a \code{factor}). This should be an unquoted column name although
this argument is passed by expression and supports
\link[rlang:quasiquotation]{quasiquotation} (you can unquote column
names). For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimate}{The column identifier for the predicted class
results (that is also \code{factor}). As with \code{truth} this can be
specified different ways but the primary method is to use an
unquoted variable name. For \verb{_vec()} functions, a \code{factor} vector.}

\item{estimator}{One of: \code{"binary"}, \code{"macro"}, \code{"macro_weighted"},
or \code{"micro"} to specify the type of averaging to be done. \code{"binary"} is
only relevant for the two class case. The other three are general methods for
calculating multiclass metrics. The default will automatically choose \code{"binary"}
or \code{"macro"} based on \code{estimate}.}

\item{na_rm}{A \code{logical} value indicating whether \code{NA}
values should be stripped before the computation proceeds.}
}
\value{
A \code{tibble} with columns \code{.metric}, \code{.estimator},
and \code{.estimate} and 1 row of values.

For grouped data frames, the number of rows returned will be the same as
the number of groups.

For \code{precision_vec()}, a single \code{numeric} value (or \code{NA}).
}
\description{
These functions calculate the \code{\link[=precision]{precision()}} of a measurement system for
finding relevant documents compared to reference results
(the truth regarding relevance). Highly related functions are \code{\link[=recall]{recall()}}
and \code{\link[=f_meas]{f_meas()}}.
}
\details{
The precision is the percentage of predicted truly relevant results
of the total number of predicted relevant results and
characterizes the "purity in retrieval performance" (Buckland
and Gey, 1994).

When the denominator of the calculation is \code{0}, precision is undefined. This
happens when both \verb{# true_positive = 0} and \verb{# false_positive = 0} are true,
which mean that there were no predicted events. When computing binary
precision, a \code{NA} value will be returned with a warning. When computing
multiclass precision, the individual \code{NA} values will be removed, and the
computation will procede, with a warning.
}
\section{Relevant Level}{


There is no common convention on which factor level should
automatically be considered the "event" or "positive" result.
In \code{yardstick}, the default is to use the \emph{first} level. To
change this, a global option called \code{yardstick.event_first} is
set to \code{TRUE} when the package is loaded. This can be changed
to \code{FALSE} if the \emph{last} level of the factor is considered the
level of interest by running: \code{options(yardstick.event_first = FALSE)}.
For multiclass extensions involving one-vs-all
comparisons (such as macro averaging), this option is ignored and
the "one" level is always the relevant result.
}

\section{Multiclass}{


Macro, micro, and macro-weighted averaging is available for this metric.
The default is to select macro averaging if a \code{truth} factor with more
than 2 levels is provided. Otherwise, a standard binary calculation is done.
See \code{vignette("multiclass", "yardstick")} for more information.
}

\section{Implementation}{


Suppose a 2x2 table with notation:

\tabular{rcc}{ \tab Reference \tab \cr Predicted \tab Relevant \tab
Irrelevant \cr Relevant \tab A \tab B \cr Irrelevant \tab C \tab D \cr }

The formulas used here are:

\deqn{recall = A/(A+C)}
\deqn{precision = A/(A+B)}
\deqn{F_meas_\beta = (1+\beta^2) * precision * recall/((\beta^2 * precision)+recall)}

See the references for discussions of the statistics.
}

\examples{
# Two class
data("two_class_example")
precision(two_class_example, truth, predicted)

# Multiclass
library(dplyr)
data(hpc_cv)

hpc_cv \%>\%
  filter(Resample == "Fold01") \%>\%
  precision(obs, pred)

# Groups are respected
hpc_cv \%>\%
  group_by(Resample) \%>\%
  precision(obs, pred)

# Weighted macro averaging
hpc_cv \%>\%
  group_by(Resample) \%>\%
  precision(obs, pred, estimator = "macro_weighted")

# Vector version
precision_vec(two_class_example$truth, two_class_example$predicted)

# Making Class2 the "relevant" level
options(yardstick.event_first = FALSE)
precision_vec(two_class_example$truth, two_class_example$predicted)
options(yardstick.event_first = TRUE)

}
\references{
Buckland, M., & Gey, F. (1994). The relationship
between Recall and Precision. \emph{Journal of the American Society
for Information Science}, 45(1), 12-19.

Powers, D. (2007). Evaluation: From Precision, Recall and F
Factor to ROC, Informedness, Markedness and Correlation.
Technical Report SIE-07-001, Flinders University
}
\seealso{
Other class metrics: 
\code{\link{accuracy}()},
\code{\link{bal_accuracy}()},
\code{\link{detection_prevalence}()},
\code{\link{f_meas}()},
\code{\link{j_index}()},
\code{\link{kap}()},
\code{\link{mcc}()},
\code{\link{npv}()},
\code{\link{ppv}()},
\code{\link{recall}()},
\code{\link{sens}()},
\code{\link{spec}()}

Other relevance metrics: 
\code{\link{f_meas}()},
\code{\link{recall}()}
}
\author{
Max Kuhn
}
\concept{class metrics}
\concept{relevance metrics}
