% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/logrr.R
\name{logrr}
\alias{logrr}
\title{Log ratio of spatial densities}
\usage{
logrr(
  x,
  sigma = NULL,
  sigmacon = NULL,
  case = 2,
  nsim = 0,
  level = 0.9,
  alternative = "two.sided",
  ...,
  bwargs = list(),
  weights = NULL,
  edge = TRUE,
  varcov = NULL,
  at = "pixels",
  leaveoneout = TRUE,
  adjust = 1,
  diggle = FALSE,
  kernel = "gaussian",
  scalekernel = is.character(kernel),
  positive = FALSE,
  verbose = TRUE
)
}
\arguments{
\item{x}{A \code{\link[spatstat.geom]{ppp}} object 
package with marks for the case and control groups.
\code{x$marks} is assumed to be a factor.  Automatic 
conversion is attempted if it is not.}

\item{sigma}{Standard deviation of isotropic smoothing kernel for cases.
Either a numerical value, or a function that computes an appropriate value
of \code{sigma}. If not specified, then
\code{\link[spatstat.core]{bw.relrisk}} is used.}

\item{sigmacon}{Standard deviation of isotropic smoothing kernel for
controls.  Default is the same as \code{sigma}.}

\item{case}{The name of the desired "case" group in \code{levels(x$marks)}.
Alternatively, the position of the name of the "case" group in
\code{levels(x$marks)}.  Since we don't know the group names, the default
is 2, the second position of \code{levels(x$marks)}. \code{x$marks} is
assumed to be a factor.  Automatic conversion is attempted if it is not.}

\item{nsim}{The number of simulated data sets from which to construct
tolerance envelopes under the random labeling hypothesis.  The default is 0
(i.e., no envelopes).}

\item{level}{The level of the tolerance envelopes.}

\item{alternative}{The type of envelopes to construct.  The default is
\code{"two.sided"} (upper and lower envelopes).  The values \code{"less"}
(lower envelope) and \code{"greater"} (upper envelope) are also valid.}

\item{...}{
    Additional arguments passed to \code{\link{pixellate.ppp}}
    and \code{\link{as.mask}} to determine
    the pixel resolution, or passed to \code{sigma} if it is a function.
  }

\item{bwargs}{A list of arguments for the bandwidth function supplied to
\code{sigma} and \code{sigmacon}, if applicable.}

\item{weights}{
    Optional weights to be attached to the points.
    A numeric vector, numeric matrix, an \code{expression},
    or a pixel image.
  }

\item{edge}{
    Logical value indicating whether to apply edge correction.
  }

\item{varcov}{
    Variance-covariance matrix of anisotropic smoothing kernel.
    Incompatible with \code{sigma}.
  }

\item{at}{
    String specifying whether to compute the intensity values
    at a grid of pixel locations (\code{at="pixels"}) or
    only at the points of \code{x} (\code{at="points"}).
  }

\item{leaveoneout}{
    Logical value indicating whether to compute a leave-one-out
    estimator. Applicable only when \code{at="points"}.
  }

\item{adjust}{
    Optional. Adjustment factor for the smoothing parameter.
  }

\item{diggle}{
    Logical. If \code{TRUE}, use the Jones-Diggle improved edge correction,
    which is more accurate but slower to compute than the default
    correction.
  }

\item{kernel}{
    The smoothing kernel.
    A character string specifying the smoothing kernel
    (current options are \code{"gaussian"}, \code{"epanechnikov"},
    \code{"quartic"} or \code{"disc"}),
    or a pixel image (object of class \code{"im"})
    containing values of the kernel, or a \code{function(x,y)} which
    yields values of the kernel.
  }

\item{scalekernel}{
    Logical value.
    If \code{scalekernel=TRUE}, then the kernel will be rescaled
    to the bandwidth determined by \code{sigma} and \code{varcov}:
    this is the default behaviour when \code{kernel} is a character string.
    If \code{scalekernel=FALSE}, then \code{sigma} and \code{varcov}
    will be ignored: this is the default behaviour when \code{kernel} is a
    function or a pixel image.
  }

\item{positive}{
    Logical value indicating whether to force all density values to
    be positive numbers. Default is \code{FALSE}.
  }

\item{verbose}{
    Logical value indicating whether to issue warnings
    about numerical problems and conditions.
  }
}
\value{
The function produces an object of type \code{logrrenv}.  Its
  components are similar to those returned by the \code{density.ppp} function
  from the \code{spatstat.core} package, with the intensity values replaced
  by the log ratio of spatial densities of f and g.  Includes an array
  \code{simr} of dimension c(nx, ny, nsim + 1), where nx and ny are the
  number of x and y grid points used to estimate the spatial density.
  \code{simr[,,1]} is the log ratio of spatial densities for the observed
  data, and the remaining \code{nsim} elements in the third dimension of the
  array are the log ratios of spatial densities from a new ppp simulated
  under the random labeling hypothesis.
}
\description{
\code{logrr} computes the estimated log relative risk of cases relative to
controls. The log relative risk at location s is defined as \code{r(s) =
ln(f(s)/g(s))}. The numerator, \code{f(s)}, is the spatial density of the
case group. The denominator, \code{g(s)}, is the spatial density of the
control group. If \code{nsim > 0}, then pointwise (at each pixel) tolerance
envelopes are estimated under the random labeling hypothesis. The tolerance
envelopes can be used to assess pixels where the log relative risk differs
significantly from zero. See Details.
}
\details{
If \code{nsim=0}, the \code{plot} function creates a heat map of the log
relative risk. If \code{nsim > 0}, the \code{plot} function colors the pixels
where the estimated log relative risk is outside the tolerance envelopes
created under the random labeling hypothesis (i.e., pixels with potential
clustering of cases or controls). Colored regions with values above 0
indicate a cluster of cases relative to controls (without controlling for
multiple comparisons), i.e., a region where the the density of the cases is
greater than the the density of the controls. Colored regions with values
below 0 indicate a cluster of controls relative to cases (without controlling
for multiple comparisons), i.e., a region where the density of the controls
is greater than the density of the cases.

The \code{two.sided} alternative test constructs two-sided tolerance
envelopes to assess whether the estimated \code{r(s)} deviates more than what
is expected under the random labeling hypothesis.  The \code{greater}
alternative constructs an upper tolerance envelope to assess whether the
estimated \code{r(s)} is greater than what is expected under the random
labeling hypothesis, i.e., where there is clustering of cases relative to
controls. The \code{lower} alternative constructs a lower tolerance envelope
to assess whether the estimated \code{r(s)} is lower than what is expected
under the random labeling hypothesis, i.e., where there is clustering of
controls relative to cases.

If the estimated density of the case or control group becomes too small, this
function may produce warnings due to numerical underflow. Increasing the
bandwidth (\code{sigma}) may help.
}
\examples{
data(grave)
# estimate and plot log relative risk
r = logrr(grave, case = "affected")
plot(r)
# use scott's bandwidth
r2 = logrr(grave, case = 2, sigma = spatstat.core::bw.scott)
plot(r2)
# construct pointwise tolerance envelopes for log relative risk
renv = logrr(grave, nsim = 9)
print(renv) # print information about envelopes
plot(renv) # plot results
# plot using a better gradient
grad = gradient.color.scale(min(renv$v, na.rm = TRUE), max(renv$v, na.rm = TRUE))
plot(renv, col = grad$col, breaks = grad$breaks, conlist = list(col = "lightgrey"))
}
\references{
Waller, L.A. and Gotway, C.A. (2005). Applied Spatial Statistics
  for Public Health Data. Hoboken, NJ: Wiley.

  Kelsall, Julia E., and Peter J. Diggle. "Kernel estimation of relative
  risk." Bernoulli (1995): 3-16.

  Kelsall, Julia E., and Peter J. Diggle. "Non-parametric estimation of
  spatial variation in relative risk." Statistics in Medicine 14.21-22
  (1995): 2335-2342.
}
\author{
Joshua French (and a small chunk by the authors of the
  \code{\link[spatstat.core]{density.ppp}}) function for consistency with the
  default behavior of that function).
}
