
##' Match taxonomic names to the Open Tree Taxonomy.
##'
##' Accepts one or more taxonomic names and returns information about
##' potential matches for these names to known taxa in the Open Tree
##' Taxononmy.
##'
##' This service uses taxonomic contexts to disambiguate homonyms and
##' misspelled names; a context may be specified using the
##' \code{context_name} argument. If no context is specified, then the
##' context will be inferred (i.e., the shallowest taxonomic context
##' that contains all unambiguous names in the input). Taxonomic
##' contexts are uncontested higher taxa that have been selected to
##' allow limits to be applied to the scope of TNRS searches
##' (e.g. 'match names only within flowering plants'). Once a context
##' has been identified (either user-specified or inferred), all taxon
##' name matches will performed only against taxa within that
##' context. For a list of available taxonomic contexts, see
##' \code{\link{tnrs_contexts}}.
##'
##' A name is considered unambiguous if it is not a synonym and has
##' only one exact match to any taxon name in the entire taxonomy.
##'
##' Several functions listed in the \sQuote{See also} section can be
##' used to inspect and manipulate the object generated by this
##' function.
##'
##'
##' @title Match names to the Open Tree Taxonomy
##' @param names taxon names to be queried (character vector)
##' @param context_name name of the taxonomic context to be searched
##'     (length-one character vector). Must match (case sensitive) one
##'     of the values returned by \code{\link{tnrs_contexts}}.
##' @param do_approximate_matching A logical indicating whether or not
##'     to perform approximate string (a.k.a. \dQuote{fuzzy})
##'     matching. Using \code{FALSE} will greatly improve
##'     speed. Default, however, is \code{TRUE}.
##' @param ids An array of ids to use for identifying names. These
##'     will be assigned to each name in the names array. If ids is
##'     provided, then ids and names must be identical in length.
##' @param include_deprecated A boolean indicating whether or not to
##'     include deprecated taxa in the search.
##' @param include_dubious Whether to include so-called 'dubious'
##'     taxa--those which are not accepted by OTT.
##' @param ...  additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @return A data frame summarizing the results of the query. The
##'     original query output is appended as an attribute to the
##'     returned object (and can be obtained using \code{attr(object,
##'     "original_response")}).
##' @seealso \code{\link{inspect.match_names}},
##'     \code{\link{update.match_names}},
##'     \code{\link{synonyms.match_names}}.
##' @examples \dontrun{
##'  deuterostomes <- tnrs_match_names(names=c("echinodermata", "xenacoelomorpha",
##'                                             "chordata", "hemichordata"))
##' }
##' @importFrom stats setNames
##' @export
tnrs_match_names <- function(names = NULL, context_name = NULL,
                             do_approximate_matching = TRUE,
                             ids = NULL, include_deprecated = FALSE,
                             include_dubious = FALSE, ...) {

    if (!is.null(context_name) &&
        !context_name %in% unlist(tnrs_contexts(...))) {
        stop("The \'context_name\' is not valid. Check possible values using tnrs_contexts()")
    }

    res <- .tnrs_match_names(names = names, context_name = context_name,
                             do_approximate_matching = do_approximate_matching,
                             ids = ids, include_deprecated = include_deprecated,
                             include_dubious = include_dubious, ...)

    check_tnrs(res)
    summary_match <- build_summary_match(res, res_id = seq_along(res[["results"]]),
                                         match_id = 1)

    summary_match$search_string <- gsub("\\\\", "", summary_match$search_string)
    summary_match <- summary_match[match(tolower(names), summary_match$search_string), ]

    summary_match[["approximate_match"]] <- convert_to_logical(summary_match[["approximate_match"]])
    summary_match[["is_synonym"]] <- convert_to_logical(summary_match[["is_synonym"]])
    summary_match[["is_deprecated"]] <- convert_to_logical(summary_match[["is_deprecated"]])

    attr(summary_match, "original_order") <- as.numeric(rownames(summary_match))
    rownames(summary_match) <- NULL
    attr(summary_match, "original_response") <- res
    attr(summary_match, "match_id") <- rep(1, nrow(summary_match))
    class(summary_match) <- c("match_names", "data.frame")
    summary_match
}

##' @importFrom stats na.omit
convert_to_logical <- function(x) {
    if (all(stats::na.omit(x) %in% c("TRUE", "FALSE"))) {
        x <- as.logical(x)
    } else {
        x
    }
}

check_tnrs <- function(req) {
    if (length(req$results) < 1) {
        stop("No matches for any of the provided taxa")
    }
    if (length(req$unmatched_name_ids)) {
        warning(paste(req$unmatched_name_ids, collapse=", "), " are not matched")
    }
}

tnrs_columns <- c("search_string" = "search_string",
                  "unique_name" = "unique_name",
                  "approximate_match" = "is_approximate_match",
                  "ott_id" = "ot:ottId",
                  "is_synonym" = "is_synonym",
                  "is_deprecated" = "is_deprecated")

summary_row_factory <- function(res, res_id, match_id, columns = tnrs_columns) {
    res_address <- res[["results"]][[res_id]][["matches"]][[match_id]]
    ret <- sapply(columns, function(cols) res_address[[cols]])
    nMatch <- length(res[["results"]][[res_id]][["matches"]])
    c(ret, number_matches = nMatch)
}

build_summary_match <- function(res, res_id, match_id = NULL) {

    if (length(res_id) > 1 &&
       (!is.null(match_id) && length(match_id) > 1)) {
        stop("Something is wrong. Please contact us.")
    }

    build_summary_row <- lapply(res_id, function(rid) {
        if (is.null(match_id)) {
            match_id <- seq_len(length(res[["results"]][[rid]][["matches"]]))
        }
        res <- lapply(match_id, function(mid) {
            summary_row_factory(res, rid, mid)
        })
        if (identical(length(match_id), 1L)) {
            unlist(res)
        } else res
    })

    if (identical(length(res_id), 1L)) {
        build_summary_row <- unlist(build_summary_row,
                                    recursive = FALSE)
    }

    ## Needed if only 1 row returned
    if (!inherits(build_summary_row, "list")) {
        build_summary_row <- list(build_summary_row)
    }

    ## Add potential unmatched names
    if (length(res[["unmatched_name_ids"]])) {
        no_match <- lapply(res[["unmatched_name_ids"]], function(x) {
            no_match_row <- stats::setNames(rep(NA, length(tnrs_columns) + 1),
                                     c(tnrs_columns, "number_matches"))
            no_match_row[1] <- x
            no_match_row
        })
        build_summary_row <- c(build_summary_row, no_match)
    }

    summary_match <- do.call("rbind", build_summary_row)
    summary_match <- data.frame(summary_match, stringsAsFactors=FALSE)
    names(summary_match) <- c(names(tnrs_columns), "number_matches")
    summary_match
}

##' This function returns a list of pre-defined taxonomic contexts
##' (i.e. clades) which can be used to limit the scope of tnrs
##' queries.
##'
##' Taxonomic contexts are available to limit the scope of TNRS
##' searches. These contexts correspond to uncontested higher taxa
##' such as 'Animals' or 'Land plants'. This service returns a list
##' containing all available taxonomic context names, which may be
##' used as input (via the \code{context_name} argument in other
##' functions) to limit the search scope of other services including
##' \code{\link{tnrs_match_names}}.
##' @title TNRS contexts
##' @param ...  additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @return Returns invisibly a list for each major clades (e.g.,
##'     animals, microbes, plants, fungi, life) whose elements
##'     contains the possible contexts.
##' @export

tnrs_contexts <- function(...) {
    res <- .tnrs_contexts(...)
    class(res) <- "tnrs_contexts"
    res
}

##' @export
print.tnrs_contexts <- function(x, ...) {
    cat("Possible contexts:\n")
    lapply(x, function(t) {
        res <- unlist(t)
        cat("  ", res[1], "\n")
        if (length(res) > 1)
            lapply(seq(2, length(t), by = 5), function(l) {
                m <- ifelse(l + 5 <= length(t), l+5, length(t))
                cat("     ", paste(t[l:m], collapse = ", "), "\n")
            })
    })
}

##' Return a taxonomic context given a list of taxonomic names
##'
##' Find the least inclusive taxonomic context that includes all the
##' unambiguous names in the input set. Unambiguous names are names
##' with exact matches to non-homonym taxa. Ambiguous names (those
##' without exact matches to non-homonym taxa) are indicated in
##' results.
##'
##' @title Infer the taxonomic context from a list of names
##' @param names Vector of taxon names.
##' @param ...  additional arguments to customize the API request (see
##'     \code{\link{rotl}} package documentation).
##' @return A list including the context name, the context ott id and
##'     possibly the names in the query that have an ambiguous
##'     taxonomic meaning in the query.
##' @examples
##' \dontrun{
##' res <- tnrs_infer_context(names=c("Stellula calliope", "Struthio camelus"))
##' }
##' @export
tnrs_infer_context <- function(names=NULL, ...) {
    res <- .tnrs_infer_context(names = names, ...)
    return(res)
}
