% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pid_dta.R
\name{pid.cvm}
\alias{pid.cvm}
\title{Independence-based identification of panel SVAR models via Cramer-von Mises (CVM) distance}
\usage{
pid.cvm(
  x,
  combine = c("group", "pool", "indiv"),
  n.factors = NULL,
  dd = NULL,
  itermax = 500,
  steptol = 100,
  iter2 = 75
)
}
\arguments{
\item{x}{An object of class '\code{pvarx}' or a list of VAR objects 
that will be \link[=as.varx]{coerced} to '\code{varx}'. 
Estimated panel of VAR objects.}

\item{combine}{Character. The combination of the individual reduced-form residuals 
via '\code{group}' for the group ICA by Calhoun et al. (2001) using common structural shocks,
via '\code{pool}' for the pooled shocks by Herwartz and Wang (2024) using a common rotation matrix, or
via '\code{indiv}' for individual-specific \eqn{B_i \forall i} using strictly separated identification runs.}

\item{n.factors}{Integer. Number of common structural shocks across all individuals if the group ICA is selected.}

\item{dd}{Object of class '\code{indepTestDist}' generated by '\code{indepTest}' from package '\code{copula}'. 
Simulated independent sample(s) of the same size as the data. 
If the sample sizes \eqn{T_i - p_i} differ across '\eqn{i}', the strictly separated identification
requires a list of \eqn{N} individual '\code{indepTestDist}'-objects with respective sample sizes.
If \code{NULL} (the default), a suitable object will be calculated during the call of \code{\link{pid.cvm}}.}

\item{itermax}{Integer. Maximum number of iterations for DEoptim.}

\item{steptol}{Numeric. Tolerance for steps without improvement for DEoptim.}

\item{iter2}{Integer. Number of iterations for the second optimization.}
}
\value{
List of class '\code{pid}' with elements:
\item{A}{Matrix. The lined-up coefficient matrices \eqn{A_j, j=1,\ldots,p} 
   for the lagged variables in the panel VAR.}
\item{B}{Matrix. Mean group of the estimated structural impact matrices \eqn{B_i}, 
   i.e. the unique decomposition of the covariance matrices of reduced-form errors.}
\item{L.varx}{List of '\code{varx}' objects for the individual estimation results 
  to which the structural impact matrices \eqn{B_i} have been added.}
\item{eps0}{Matrix. The combined whitened residuals \eqn{\epsilon_{0}} 
  to which the ICA procedure has been applied subsequently. 
  These are still the unrotated baseline shocks! 
  \code{NULL} if '\code{indiv}' identifications have been used.}
\item{ICA}{List of objects resulting from the underlying ICA procedure. 
  \code{NULL} if '\code{indiv}' identifications have been used.}
\item{rotation_angles}{Numeric vector. The rotation angles 
  suggested by the combined identification procedure. 
  \code{NULL} if '\code{indiv}' identifications have been used.}
\item{args_pid}{List of characters and integers indicating the identification methods and specifications that have been used.}
\item{args_pvarx}{List of characters and integers indicating the estimator and specifications that have been used.}
}
\description{
Given an estimated panel of VAR models, this function applies independence-based identification for 
  the structural impact matrix \eqn{B_i} of the corresponding SVAR model
  \deqn{y_{it} = c_{it} + A_{i1} y_{i,t-1} + ... + A_{i,p_i} y_{i,t-p_i} + u_{it}}
  \deqn{       = c_{it} + A_{i1} y_{i,t-1} + ... + A_{i,p_i} y_{i,t-p_i} + B_i \epsilon_{it}.}
  Matrix \eqn{B_i} corresponds to the unique decomposition of the least squares covariance matrix \eqn{\Sigma_{u,i} = B_i B_i'} 
  if the vector of structural shocks \eqn{\epsilon_{it}} contains at most one Gaussian shock (Comon, 1994).
  A nonparametric dependence measure, the Cramer-von Mises distance (Genest and Remillard, 2004), 
  determines least dependent structural shocks. The minimum is obtained by a two step optimization algorithm 
  similar to the technique described in Herwartz and Ploedt (2016).
}
\examples{
\donttest{
# select minimal or full example #
is_min = TRUE
idx_i  = ifelse(is_min, 1, 1:14)

# load and prepare data #
data("EURO")
data("EU_w")
names_i = names(EURO[idx_i+1])  # country names (#1 is EA-wide aggregated data)
idx_k   = 1:4   # endogenous variables in individual data matrices
idx_t   = 1:76  # periods from 2001Q1 to 2019Q4
trend2  = idx_t^2

# individual VARX models with common lag-order p=2 #
L.data = lapply(EURO[idx_i+1], FUN=function(x) x[idx_t, idx_k])
L.exog = lapply(EURO[idx_i+1], FUN=function(x) cbind(trend2, x[idx_t, 5:10]))
L.vars = sapply(names_i, FUN=function(i) 
  vars::VAR(L.data[[i]], p=2, type="both", exogen=L.exog[[i]]), 
  simplify=FALSE)

# identify under common orthogonal matrix (with pooled sample size (T-p)*N) #
S.pind = copula::indepTestSim(n=(76-2)*length(names_i), p=length(idx_k), N=100)
R.pcvm = pid.cvm(L.vars, dd=S.pind, combine="pool")
R.irf  = irf(R.pcvm, n.ahead=50, w=EU_w)
plot(R.irf, selection=list(1:2, 3:4))

# identify individually (with same sample size T-p for all 'i') #
S.pind = copula::indepTestSim(n=(76-2), p=length(idx_k), N=100)
R.pcvm = pid.cvm(L.vars, dd=S.pind, combine="indiv")
R.irf  = irf(R.pcvm, n.ahead=50, w=EU_w)
plot(R.irf, selection=list(1:2, 3:4))
}

}
\references{
Comon, P. (1994): 
  "Independent Component Analysis: A new Concept?", 
  \emph{Signal Processing}, 36, pp. 287-314.

Genest, C., and Remillard, B. (2004): 
  "Tests of Independence and Randomness based on the Empirical Copula Process", 
  Test, 13, pp. 335-370.

Herwartz, H., and Wang, S. (2024): 
  "Statistical Identification in Panel Structural Vector Autoregressive 
  Models based on Independence Criteria", 
  \emph{Journal of Applied Econometrics}, 39 (4), pp. 620-639.

Herwartz, H. (2018): 
  "Hodges Lehmann detection of structural shocks - An Analysis of macroeconomic Dynamics in the Euro Area", 
  \emph{Oxford Bulletin of Economics and Statistics}, 80, pp. 736-754.

Herwartz, H., and Ploedt, M. (2016): 
  "The Macroeconomic Effects of Oil Price Shocks: Evidence from a Statistical Identification Approach", 
  \emph{Journal of International Money and Finance}, 61, pp. 30-44.
}
\seealso{
\ldots the individual \code{\link[svars]{id.cvm}} by Lange et al. (2021) in \strong{svars}. 
  Note that \code{\link{pid.cvm}} relies on a modification of their procedure and 
  thus performs ICA on the pre-whitened shocks '\code{eps0}' directly.

Other panel identification functions: 
\code{\link{pid.chol}()},
\code{\link{pid.dc}()},
\code{\link{pid.grt}()},
\code{\link{pid.iv}()}
}
\concept{panel identification functions}
