#' Create a Bounding Box
#'
#' Convencience method to create a bounding box like that returned by \code{sp::bbox()}.
#' To generate a bounding box from lists of lat/lon values use \code{sp::bbox(cbind(lons, lats))}.
#'
#' @param n North bounding latitude
#' @param e East bounding longitude
#' @param s South bounding latitude
#' @param w West bounding longitude
#' @return A 2x2 matrix describing a bounding box like that returned by \code{sp::bbox()}
#' @seealso sp::bbox
#' @examples makebbox(45.125, -64.25, 44.875, -64.75)
#'
#' @export
makebbox <- function(n, e, s, w) {
  matrix(c(w, s, e, n), byrow=FALSE, ncol=2, dimnames=list(c("x", "y"), c("min", "max")))
}

#' Query The Interwebs For A Bounding Box
#'
#' Use the \href{https://pickpoint.io/}{PickPoint.io API} to
#' retreive a bounding box for the given query. Implemented
#' from the \code{ggmap:geocode} function from the \code{ggmap}
#' package (\url{https://cran.r-project.org/package=ggmap})
#' by David Kahle to remove dependencies
#' of \code{ggmap} that are not necessary for \code{prettymapr}. Note that if
#' you would like to use \code{google} as a source, you must agree to the Google
#' API \href{https://developers.google.com/terms/}{terms and conditions}.
#'
#' @param querystring The search query.
#' @param ... Additional paramters to be passed on to \link{geocode}. Passing \code{source="google"}
#'   may be useful if google is desired as a source. Use \code{options(prettymapr.geosource="google")}
#'   to permanently use \code{google} as a source.
#' @return A 2x2 matrix describing a bounding box like that returned by \code{sp::bbox()}
#'
#' @examples
#' #don't test to speed up checking time
#' \donttest{
#' searchbbox("kings county, NS")
#' searchbbox("University Ave. Wolfville NS", source="google")
#' searchbbox("Wolfville ns", source="google")
#' searchbbox(c("Vermont", "Nova Scotia"))
#' }
#'
#' @export
#'
searchbbox <- function(querystring, ...) {

  d <- geocode(querystring, output="data.frame", limit=1, ...)
  if(nrow(d) > 1) {
    #"import" foreach
    foreach <- foreach::foreach
    `%do%` <- foreach::`%do%`
    i<-NULL;rm(i) #trick CMD check
    foreach(i=1:nrow(d)) %do% {
      makebbox(d$bbox_n[i], d$bbox_e[i], d$bbox_s[i], d$bbox_w[i])
    }
  } else {
    makebbox(d$bbox_n, d$bbox_e, d$bbox_s, d$bbox_w)
  }
}

#' Zoom the extents of a bounding box
#'
#' Manipulate the extents of a bounding box by zooming and moving an
#' existing bbox. This is helpful when manipulating the extents of a
#' plot created by \code{canvec.qplot()}
#'
#' @param bbox An existing bbox
#' @param factor A factor to zoom by. >1 will zoom in, <1 will zoom out.
#' If a vector is passed, the first element will zoom the X extent, the
#' second element will zoom the Y extent.
#' @param offset A vector describing the X and Y offset that should be applied.
#' @return A zoomed bounding box.
#'
#' @examples
#' \donttest{
#' alta <- searchbbox("alta lake bc", source="google")
#' zoombbox(alta, c(.2,.5))
#' }
#' @export
#'
zoombbox <- function(bbox, factor, offset=c(0,0)) {
  lons <- bbox[1,]
  lats <- bbox[2,]
  clon <- mean(lons) + offset[1]
  clat <- mean(lats) + offset[2]

  if(length(factor)>1) {
    factorx <- factor[1]
    factory <- factor[2]
  } else {
    factorx <- factor
    factory <- factor
  }

  newwidth <- (lons[2]-lons[1]) / factorx
  newheight <- (lats[2]-lats[1]) / factory

  makebbox(clat+newheight/2.0, clon+newwidth/2.0, clat-newheight/2.0, clon-newwidth/2.0)
}
