% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ps_rand.R
\name{ps_rand}
\alias{ps_rand}
\title{Null model randomization analysis of alpha diversity metrics}
\usage{
ps_rand(
  ps,
  metric = c("PD", "PE", "RPE", "CE"),
  fun = "tip_shuffle",
  method = NULL,
  n_rand = 100,
  summary = "quantile",
  spatial = TRUE,
  n_cores = 1,
  progress = interactive(),
  ...
)
}
\arguments{
\item{ps}{\code{phylospatial} object.}

\item{metric}{Character vector giving one or more diversity metrics to calculate; see \link{ps_diversity}
for options. Can also specify \code{"all"} to calculate all available metrics.}

\item{fun}{Null model function to use. Must be either "tip_shuffle", "nullmodel", "quantize", or an actual function:
\itemize{
\item "tip_shuffle" (the default): randomly shuffles the identities of terminal taxa
\item "nullmodel": uses \link[vegan]{nullmodel} and \link[vegan]{simulate.nullmodel}, from the vegan
package, which offer a wide range of randomization algorithms with different properties.
\item "quantize": uses \link[nullcat]{quantize}, which converts a quantitative matrix to discrete strata,
applies a categorical variant of the selected null model, and then maps randomized strata back to values.
\item Any other function that accepts a community matrix as its first argument and returns a
randomized version of the matrix.
}}

\item{method}{Method used by the selected function.
\itemize{
\item For \code{fun = "nullmodel"}, one of the method options listed under \link[vegan]{commsim}.
Be sure to select a method that is appropriate to your community \code{data_type} (binary, quantitative, abundance),
\item For \code{fun = "quantize"}, one of the categorical algorithms listed under \link[nullcat]{nullcat}.
\item Ignored if \code{fun} is \code{"tip_shuffle"} or if it is a custom function.
}}

\item{n_rand}{Integer giving the number of random communities to generate.}

\item{summary}{Character indicating which summary statistic to return. If \code{"quantile"}, the default, the function
returns the proportion of randomizations in which the observed diversity metric was greater than the randomized
metric. If \code{"zscore"}, it returns a "standardized effect size" or z-score relating the observed value to the mean and
standard deviation of the randomizations.}

\item{spatial}{Logical: should the function return a spatial object (TRUE, default) or a matrix (FALSE).}

\item{n_cores}{Integer giving the number of compute cores to use for parallel processing.}

\item{progress}{Logical: should a progress bar be displayed?}

\item{...}{Additional arguments passed to \link[nullcat]{quantize}, \link[vegan]{simulate.nullmodel}, or custom function
\code{fun}. Note that the \code{nsim} argument to simulate.nullmodel should not be used here; specify \code{n_rand} instead.}
}
\value{
A matrix with a row for every row of \code{x}, a column for every metric specified in \code{metric}, and
values for the \code{summary} statistic. Or if \code{spatial = TRUE}, a \code{sf} or \code{SpatRaster} object containing these data.
}
\description{
This function compares phylodiversity metrics calculated in \link{ps_diversity} to their null distributions
computed by randomizing the community matrix or shuffling the tips of the phylogeny, indicating statistical
significance under the assumptions of the null model. Various null model algorithms are available for
binary, probability, and count data.
}
\examples{
\donttest{
# simulate a `phylospatial` data set and run randomization with default settings
ps <- ps_simulate(data_type = "prob")
rand <- ps_rand(ps)

# using the default `tip_shuffle` function, but with alternative arguments
rand <- ps_rand(ps, transform = sqrt, n_strata = 4, priority = "rows")

# using the `quantize` function with the `curvecat` algorithm
if(requireNamespace("nullcat")){
    rand <- ps_rand(ps,
      fun = "quantize", method = "curvecat",
      transform = sqrt, n_strata = 4, fixed = "cell")
}

# using binary data, with a vegan `nullmodel` algorithm
ps2 <- ps_simulate(data_type = "binary")
rand <- ps_rand(ps2, fun = "nullmodel", method = "r2")

# using abundance data, and demonstrating alternative metric choices
ps3 <- ps_simulate(data_type = "abund")
rand <- ps_rand(ps3, metric = c("ShPD", "SiPD"),
      fun = "nullmodel", method = "abuswap_c")
rand
}
}
\seealso{
\code{\link[=ps_diversity]{ps_diversity()}}
}
