% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/OptimizerSuccessiveHalving.R
\name{mlr_optimizers_successive_halving}
\alias{mlr_optimizers_successive_halving}
\alias{OptimizerSuccessiveHalving}
\title{Hyperparameter Optimization with Successive Halving}
\source{
Jamieson K, Talwalkar A (2016).
\dQuote{Non-stochastic Best Arm Identification and Hyperparameter Optimization.}
In Gretton A, Robert CC (eds.), \emph{Proceedings of the 19th International Conference on Artificial Intelligence and Statistics}, volume 51 series Proceedings of Machine Learning Research, 240-248.
\url{http://proceedings.mlr.press/v51/jamieson16.html}.
}
\description{
\code{OptimizerSuccessiveHalving} class that implements the successive halving
algorithm (SHA). SHA randomly samples \code{n} candidate points and
allocates a minimum budget (\code{r_min}) to all candidates. The candidates are
raced down in stages to a single best candidate by repeatedly increasing the
budget by a factor of \code{eta} and promoting only the best \code{1 / eta } fraction
to the next stage. This means promising points are allocated a higher budget
overall and lower performing ones are discarded early on.

#' The budget hyperparameter must be tagged with \code{"budget"} in the search space.
The minimum budget (\code{r_min}) which is allocated in the base stage, is set by
the lower bound of the budget parameter. The upper bound  defines the maximum
budget (\code{r_max}) which is allocated to the candidates in the last stage. The
number of stages is computed so that each candidate in base stage is
allocated the minimum budget and the candidates in the last stage are not
evaluated on more than the maximum budget. The following table is the stage
layout for \code{eta = 2}, \code{r_min = 1} and \code{r_max = 8}.\tabular{rrr}{
   \code{i} \tab \code{n_i} \tab \code{r_i} \cr
   0 \tab 8 \tab 1 \cr
   1 \tab 4 \tab 2 \cr
   2 \tab 2 \tab 4 \cr
   3 \tab 1 \tab 8 \cr
}


\code{i} is stage number, \code{n_i} is the number of configurations and \code{r_i} is the
budget allocated to a single configuration.
}
\section{Parameters}{

\describe{
\item{\code{n}}{\code{integer(1)}\cr
Number of points in base stage.}
\item{\code{eta}}{\code{numeric(1)}\cr
With every stage, the budget is increased by a factor of \code{eta}
and only the best \code{1 / eta} points are promoted to the next stage.
}
\item{\code{sampler}}{\link[paradox:Sampler]{paradox::Sampler}\cr
Object defining how the samples of the parameter space should be drawn. The
default is uniform sampling.
}
\item{\code{repetitions}}{\code{integer(1)}\cr
If \code{1} (default), optimization is stopped once all stages are evaluated.
Otherwise, optimization is stopped after \code{repetitions} runs of SHA. The
\link[bbotk:Terminator]{bbotk::Terminator} might stop the optimization before all repetitions are
executed.
}
\item{\code{adjust_minimum_budget}}{\code{logical(1)}\cr
If \code{TRUE}, minimum budget is increased so that the last stage uses the
maximum budget defined in the search space.
}}
}

\section{Archive}{

The \link[bbotk:Archive]{bbotk::Archive} holds the following additional columns that are specific
to the successive halving algorithm:
\itemize{
\item \code{stage} (\verb{integer(1))}\cr
Stage index. Starts counting at 0.
\item \code{repetition} (\verb{integer(1))}\cr
Repetition index. Start counting at 1.
}
}

\section{Custom Sampler}{

Hyperband supports custom \link[paradox:Sampler]{paradox::Sampler} object for initial
configurations in each bracket.
A custom sampler may look like this (the full example is given in the
\emph{examples} section):

\if{html}{\out{<div class="sourceCode">}}\preformatted{# - beta distribution with alpha = 2 and beta = 5
# - categorical distribution with custom probabilities
sampler = SamplerJointIndep$new(list(
  Sampler1DRfun$new(params[[2]], function(n) rbeta(n, 2, 5)),
  Sampler1DCateg$new(params[[3]], prob = c(0.2, 0.3, 0.5))
))
}\if{html}{\out{</div>}}
}

\section{Progress Bars}{

\verb{$optimize()} supports progress bars via the package \CRANpkg{progressr}
combined with a \link{Terminator}. Simply wrap the function in
\code{progressr::with_progress()} to enable them. We recommend to use package
\CRANpkg{progress} as backend; enable with \code{progressr::handlers("progress")}.
}

\section{Logging}{

Hyperband uses a logger (as implemented in \CRANpkg{lgr}) from package
\CRANpkg{bbotk}.
Use \code{lgr::get_logger("bbotk")} to access and control the logger.
}

\examples{
library(bbotk)
library(data.table)

# set search space
search_space = domain = ps(
  x1 = p_dbl(-5, 10),
  x2 = p_dbl(0, 15),
  fidelity = p_dbl(1e-2, 1, tags = "budget")
)

# objective with modified branin function, see `bbotk::branin()`
objective = ObjectiveRFun$new(
  fun = branin,
  domain = domain,
  codomain = ps(y = p_dbl(tags = "minimize"))
)

# initialize instance and optimizer
instance = OptimInstanceSingleCrit$new(
  objective = objective,
  search_space = search_space,
  terminator = trm("evals", n_evals = 50)
)

optimizer = opt("successive_halving")

# optimize branin function
optimizer$optimize(instance)

# best scoring evaluation
instance$result

# all evaluations
as.data.table(instance$archive)
}
\section{Super class}{
\code{\link[bbotk:Optimizer]{bbotk::Optimizer}} -> \code{OptimizerSuccessiveHalving}
}
\section{Methods}{
\subsection{Public methods}{
\itemize{
\item \href{#method-OptimizerSuccessiveHalving-new}{\code{OptimizerSuccessiveHalving$new()}}
\item \href{#method-OptimizerSuccessiveHalving-clone}{\code{OptimizerSuccessiveHalving$clone()}}
}
}
\if{html}{\out{
<details open><summary>Inherited methods</summary>
<ul>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="format"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-format'><code>bbotk::Optimizer$format()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="help"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-help'><code>bbotk::Optimizer$help()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="optimize"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-optimize'><code>bbotk::Optimizer$optimize()</code></a></span></li>
<li><span class="pkg-link" data-pkg="bbotk" data-topic="Optimizer" data-id="print"><a href='../../bbotk/html/Optimizer.html#method-Optimizer-print'><code>bbotk::Optimizer$print()</code></a></span></li>
</ul>
</details>
}}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerSuccessiveHalving-new"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerSuccessiveHalving-new}{}}}
\subsection{Method \code{new()}}{
Creates a new instance of this \link[R6:R6Class]{R6} class.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerSuccessiveHalving$new()}\if{html}{\out{</div>}}
}

}
\if{html}{\out{<hr>}}
\if{html}{\out{<a id="method-OptimizerSuccessiveHalving-clone"></a>}}
\if{latex}{\out{\hypertarget{method-OptimizerSuccessiveHalving-clone}{}}}
\subsection{Method \code{clone()}}{
The objects of this class are cloneable with this method.
\subsection{Usage}{
\if{html}{\out{<div class="r">}}\preformatted{OptimizerSuccessiveHalving$clone(deep = FALSE)}\if{html}{\out{</div>}}
}

\subsection{Arguments}{
\if{html}{\out{<div class="arguments">}}
\describe{
\item{\code{deep}}{Whether to make a deep clone.}
}
\if{html}{\out{</div>}}
}
}
}
