% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_nest.R
\name{convert_nest}
\alias{convert_nest}
\title{Convert Nested Columns Between \code{data.frame} and \code{data.table}}
\usage{
convert_nest(data, to = c("df", "dt"), nest_cols = NULL)
}
\arguments{
\item{data}{A \code{data.frame} or \code{data.table} containing nested columns}

\item{to}{A \code{character} string specifying the target format.
Options are \code{"df"} (data frame) or \code{"dt"} (data table). Defaults to \code{"df"}.}

\item{nest_cols}{A \code{character} vector of column names containing nested data.
If \code{NULL}, the function automatically detects list columns.}
}
\value{
A transformed \code{data.frame} or \code{data.table} with nested columns converted to the specified format.
}
\description{
The \code{convert_nest} function transforms a \code{data.frame} or \code{data.table} by converting nested columns
to either \code{data.frame} or \code{data.table} format while preserving the original data structure.
}
\details{
Advanced Nested Column Conversion Features:
\itemize{
\item Intelligent automatic detection of nested columns
\item Comprehensive conversion of entire data structure
\item Selective conversion of specified nested columns
\item Non-destructive transformation with data copying
}

Input Validation and Error Handling:
\itemize{
\item Validates existence of specified nested columns
\item Verifies that specified columns are actually list columns
\item Provides informative error messages for invalid inputs
\item Ensures data integrity through comprehensive checks
}

Conversion Strategies:
\enumerate{
\item Nested column identification based on \code{is.list()} detection
\item Preservation of original data integrity
\item Flexible handling of mixed data structures
\item Consistent type conversion across nested elements
}

Nested Column Handling:
\itemize{
\item Supports conversion of \code{list} columns
\item Handles \code{data.table}, \code{data.frame}, and generic \code{list} inputs
\item Maintains original column structure and order
\item Prevents in-place modification of source data
}
}
\note{
Conversion Characteristics:
\itemize{
\item Non-destructive transformation of nested columns
\item Supports flexible input and output formats
\item Intelligent type detection and conversion
\item Minimal performance overhead
}

Error Conditions:
\itemize{
\item Throws error if specified columns don't exist in the input data
\item Throws error if specified columns are not list columns
\item Provides clear error messages for troubleshooting
\item Validates input parameters before processing
}
}
\examples{
# Example 1: Create nested data structures
# Create single nested column
df_nest1 <- iris |> 
  dplyr::group_nest(Species)     # Group and nest by Species

# Create multiple nested columns
df_nest2 <- iris |>
  dplyr::group_nest(Species) |>  # Group and nest by Species
  dplyr::mutate(
    data2 = purrr::map(          # Create second nested column
      data,
      dplyr::mutate, 
      c = 2
    )
  )

# Example 2: Convert nested structures
# Convert data frame to data table
convert_nest(
  df_nest1,                      # Input nested data frame
  to = "dt"                      # Convert to data.table
)

# Convert specific nested columns
convert_nest(
  df_nest2,                      # Input nested data frame
  to = "dt",                     # Convert to data.table
  nest_cols = "data"             # Only convert 'data' column
)

# Example 3: Convert data table to data frame
dt_nest <- mintyr::w2l_nest(
  data = iris,                   # Input dataset
  cols2l = 1:2                   # Columns to nest
)
convert_nest(
  dt_nest,                       # Input nested data table
  to = "df"                      # Convert to data frame
)
}
