/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5Omodule.h" 

#include "H5private.h"   
#include "H5Eprivate.h"  
#include "H5MMprivate.h" 
#include "H5Opkg.h"      

static void *H5O__name_decode(H5F_t *f, H5O_t *open_oh, unsigned mesg_flags, unsigned *ioflags, size_t p_size,
                              const uint8_t *p);
static herr_t H5O__name_encode(H5F_t *f, bool disable_shared, size_t H5_ATTR_UNUSED p_size, uint8_t *p,
                               const void *_mesg);
static void  *H5O__name_copy(const void *_mesg, void *_dest);
static size_t H5O__name_size(const H5F_t *f, bool disable_shared, const void *_mesg);
static herr_t H5O__name_reset(void *_mesg);
static herr_t H5O__name_debug(H5F_t *f, const void *_mesg, FILE *stream, int indent, int fwidth);

const H5O_msg_class_t H5O_MSG_NAME[1] = {{
    H5O_NAME_ID,        
    "name",             
    sizeof(H5O_name_t), 
    0,                  
    H5O__name_decode,   
    H5O__name_encode,   
    H5O__name_copy,     
    H5O__name_size,     
    H5O__name_reset,    
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    NULL,               
    H5O__name_debug     
}};

static void *
H5O__name_decode(H5F_t H5_ATTR_NDEBUG_UNUSED *f, H5O_t H5_ATTR_UNUSED *open_oh,
                 unsigned H5_ATTR_UNUSED mesg_flags, unsigned H5_ATTR_UNUSED *ioflags, size_t p_size,
                 const uint8_t *p)
{
    H5O_name_t *mesg      = NULL;
    void       *ret_value = NULL;

    FUNC_ENTER_PACKAGE

    assert(f);
    assert(p);

    if (NULL == (mesg = (H5O_name_t *)H5MM_calloc(sizeof(H5O_name_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    if (NULL == (mesg->s = (char *)H5MM_strndup((const char *)p, p_size - 1)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    ret_value = mesg;

done:
    if (NULL == ret_value)
        if (mesg) {
            H5MM_xfree(mesg->s);
            H5MM_xfree(mesg);
        }

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__name_encode(H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, size_t H5_ATTR_UNUSED p_size,
                 uint8_t *p, const void *_mesg)
{
    const H5O_name_t *mesg = (const H5O_name_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(p);
    assert(mesg && mesg->s);

    
    strcpy((char *)p, mesg->s);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static void *
H5O__name_copy(const void *_mesg, void *_dest)
{
    const H5O_name_t *mesg      = (const H5O_name_t *)_mesg;
    H5O_name_t       *dest      = (H5O_name_t *)_dest;
    void             *ret_value = NULL; 

    FUNC_ENTER_PACKAGE

    
    assert(mesg);

    if (!dest && NULL == (dest = (H5O_name_t *)H5MM_calloc(sizeof(H5O_name_t))))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    *dest = *mesg;
    if (NULL == (dest->s = H5MM_xstrdup(mesg->s)))
        HGOTO_ERROR(H5E_RESOURCE, H5E_NOSPACE, NULL, "memory allocation failed");

    
    ret_value = dest;

done:
    if (NULL == ret_value)
        if (dest && NULL == _dest)
            dest = (H5O_name_t *)H5MM_xfree(dest);

    FUNC_LEAVE_NOAPI(ret_value)
} 

static size_t
H5O__name_size(const H5F_t H5_ATTR_UNUSED *f, bool H5_ATTR_UNUSED disable_shared, const void *_mesg)
{
    const H5O_name_t *mesg      = (const H5O_name_t *)_mesg;
    size_t            ret_value = 0; 

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);

    ret_value = mesg->s ? strlen(mesg->s) + 1 : 0;

    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5O__name_reset(void *_mesg)
{
    H5O_name_t *mesg = (H5O_name_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(mesg);

    
    mesg->s = (char *)H5MM_xfree(mesg->s);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5O__name_debug(H5F_t H5_ATTR_UNUSED *f, const void *_mesg, FILE *stream, int indent, int fwidth)
{
    const H5O_name_t *mesg = (const H5O_name_t *)_mesg;

    FUNC_ENTER_PACKAGE_NOERR

    
    assert(f);
    assert(mesg);
    assert(stream);
    assert(indent >= 0);
    assert(fwidth >= 0);

    Rfprintf(stream, "%*s%-*s `%s'\n", indent, "", fwidth, "Name:", mesg->s);

    FUNC_LEAVE_NOAPI(SUCCEED)
} 
