% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ggMarginal.R
\name{ggMarginal}
\alias{ggMarginal}
\title{Add marginal density/histogram to ggplot2 scatterplots}
\usage{
ggMarginal(p, data, x, y, type = c("density", "histogram", "boxplot",
  "violin", "densigram"), margins = c("both", "x", "y"), size = 5, ...,
  xparams = list(), yparams = list(), groupColour = FALSE,
  groupFill = FALSE)
}
\arguments{
\item{p}{A ggplot2 scatterplot to add marginal plots to. If \code{p} is
not provided, then all of \code{data}, \code{x}, and \code{y} must be
provided.}

\item{data}{The data.frame to use for creating the marginal plots. Optional
if \code{p} is provided and the marginal plots are reflecting the same data.}

\item{x}{The name of the variable along the x axis. Optional if \code{p} is
provided and the \code{x} aesthetic is set in the main plot.}

\item{y}{The name of the variable along the y axis. Optional if \code{p} is
provided and the \code{y} aesthetic is set in the main plot.}

\item{type}{What type of marginal plot to show. One of: [density, histogram, boxplot, violin, densigram] 
(a "densigram" is when a density plot is overlaid on a histogram).}

\item{margins}{Along which margins to show the plots. One of: [both, x, y].}

\item{size}{Integer describing the relative size of the marginal plots
compared to the main plot. A size of 5 means that the main plot is 5x wider
and 5x taller than the marginal plots.}

\item{...}{Extra parameters to pass to the marginal plots. Any parameter that
\code{geom_line()}, \code{geom_histogram()}, \code{geom_boxplot()}, or \code{geom_violin()} accepts
can be used. For example, \code{colour = "red"} can be used for any marginal plot type,
and \code{binwidth = 10} can be used for histograms.}

\item{xparams}{List of extra parameters to use only for the marginal plot along
the x axis.}

\item{yparams}{List of extra parameters to use only for the marginal plot along
the y axis.}

\item{groupColour}{If \code{TRUE}, the colour (or outline) of the marginal
plots will be grouped according to the variable mapped to \code{colour} in the
scatter plot. The variable mapped to \code{colour} in the scatter plot must
be a character or factor variable. See examples below.}

\item{groupFill}{If \code{TRUE}, the fill of the marginal
plots will be grouped according to the variable mapped to \code{colour} in the
scatter plot. The variable mapped to \code{colour} in the scatter plot must
be a character or factor variable. See examples below.}
}
\value{
An object of class \code{ggExtraPlot}. This object can be printed to show the
plots or saved using any of the typical image-saving functions (for example, using
\code{png()} or \code{pdf()}).
}
\description{
Create a ggplot2 scatterplot with marginal density plots (default) or
histograms, or add the marginal plots to an existing scatterplot.
}
\note{
The \code{grid} and \code{gtable} packages are required for this
function.

Since the \code{size} parameter is used by \code{ggMarginal}, if you want
to pass a size to the marginal plots, you cannot
use the \code{...} parameter. Instead, you must pass \code{size} to
both \code{xparams} and \code{yparams}. For example,
\code{ggMarginal(p, size = 2)} will change the size of the main vs marginal plot,
while \code{ggMarginal(p, xparams = list(size=2), yparams = list(size=2))}
will make the density plot outline thicker.
}
\examples{
\dontrun{
library(ggplot2)

# basic usage
p <- ggplot(mtcars, aes(wt, mpg)) + geom_point()
ggMarginal(p)

# using some parameters
set.seed(30)
df <- data.frame(x = rnorm(500, 50, 10), y = runif(500, 0, 50))
p2 <- ggplot(df, aes(x, y)) + geom_point()
ggMarginal(p2)
ggMarginal(p2, type = "histogram")
ggMarginal(p2, margins = "x")
ggMarginal(p2, size = 2)
ggMarginal(p2, colour = "red")
ggMarginal(p2, colour = "red", xparams = list(colour = "blue", size = 3))
ggMarginal(p2, type = "histogram", bins = 10)

# Using violin plot
ggMarginal(p2, type = "violin")

# Using a "densigram" plot
ggMarginal(p2, type = "densigram")

# specifying the data directly instead of providing a plot
ggMarginal(data = df, x = "x", y = "y")

# more examples showing how the marginal plots are properly aligned even when
# the main plot axis/margins/size/etc are changed
set.seed(30)
df2 <- data.frame(x = c(rnorm(250, 50, 10), rnorm(250, 100, 10)),
                  y = runif(500, 0, 50))
p2 <- ggplot(df2, aes(x, y)) + geom_point()
ggMarginal(p2)

p2 <- p2 + ggtitle("Random data") + theme_bw(30)
ggMarginal(p2)

p3 <- ggplot(df2, aes(log(x), y - 500)) + geom_point()
ggMarginal(p3)

p4 <- p3 + scale_x_continuous(limits = c(2, 6)) + theme_bw(50)
ggMarginal(p4)

# Using groupColour and groupFill
# In order to use either of these arguments, we must map 'colour' in the
# scatter plot to a factor or character variable
p <- ggplot(mtcars, aes(x = wt, y = drat, colour = factor(vs))) +
     geom_point()
ggMarginal(p, groupColour = TRUE)
ggMarginal(p, groupColour = TRUE, groupFill = TRUE)
}

}
\seealso{
\href{http://daattali.com/shiny/ggExtra-ggMarginal-demo/}{Demo Shiny app}
}
