#' sim2Animation: Animate BAM simulation object.
#'
#' @description Animates BAM simulation object.
#' @param sdm_simul A bam object. See \code{\link[bamm]{sdm_sim}}
#' @param which_steps A numeric vector indicating the simulation steps that
#' are going to be converted into raster layers.
#' @param extra_legend A legend  to add to the animation.
#' @param bg_color Color for unsuitable pixels. Default "#F6F2E5".
#' @param suit_color Color for suitable pixels. Default "#0076BE".
#' @param occupied_color Color for occupied pixels. Default "#03C33F".
#' @param ani.width Animation width unit in px.
#' @param ani.height Animation height unit in px.
#' @param ani.res Animation resolution unit in px.
#' @param ani.width Animation width unit in px
#' @param ani.height Animation height unit in px
#' @param ani.res Animation resolution unit in px
#' @param gif_vel A value that regulates the velocity of frame transitions.
#' The bigger it is the transition will be slower
#' default 0.8
#' @param fmt Animation format. Possible values are GIF and HTML
#' @param filename File name.
#' @param png_keyword A keyword name for the png images generated by the
#' function
#' @return A RasterStack of species' distribution at each simulation step
#' @details The animation can be saved in a GIF or HTML format. Note that
#' the generation of the GIF can be time consuming for large simulation
#' (simulations with more than 60 time steps).
#'
#' @author Luis Osorio-Olvera & Jorge Soberón
#' @export
#' @examples

#' model_path <- system.file("extdata/Lepus_californicus_cont.tif",
#'                           package = "bamm")
#' model <- raster::raster(model_path)

#' sparse_mod <- bamm::model2sparse(model,0.1)
#' adj_mod <- bamm::adj_mat(sparse_mod,ngbs=2)

#' occs_lep_cal <- data.frame(longitude = c(-115.10417,
#'                                          -104.90417),
#'                            latitude = c(29.61846,
#'                                         29.81846))

#' occs_sparse <- bamm::occs2sparse(modelsparse = sparse_mod,
#'                                 occs = occs_lep_cal)
#' sdm_lep_cal <- bamm::sdm_sim(set_A = sparse_mod,
#'                             set_M = adj_mod,
#'                             initial_points = occs_sparse,
#'                             nsteps = 50)
#' \donttest{
#' if(requireNamespace("animation")){
#' ani_name <- tempfile(pattern = "simulation_",fileext = ".html")
#' #sdm_lep_cal_st <- bamm::sim2Animation(sdm_simul = sdm_lep_cal,
#' #                                      which_steps = seq(1,50,by=1),
#' #                                      fmt = "HTML",ani.width = 1200,
#' #                                      ani.height = 1200,
#' #                                      filename = ani_name)
#' }
#' }

sim2Animation <- function(sdm_simul, which_steps,
                          fmt = "GIF", filename,
                          png_keyword = NULL,
                          extra_legend = NULL,
                          bg_color = "#F6F2E5",
                          suit_color = "#0076BE",
                          occupied_color = "#03C33F",
                          gif_vel = 0.8,
                          ani.width = 1200,
                          ani.height = 1200,
                          ani.res = 300) {

  if(is.null(png_keyword)){
    png_keyword <- gsub("[.]","_",basename(filename))
    #png_keyword <- gsub(,"_")
  }


  # Validate inputs
  fmt <- toupper(fmt)
  if(!fmt %in% c("GIF", "HTML")) {
    stop("fmt should be either 'GIF' or 'HTML'")
  }

  # Prepare directory structure
  filename <- normalizePath(filename, mustWork = FALSE)
  dir2 <- dirname(filename)

  # Prepare animation steps and titles
  which_steps <- c(0, which_steps)
  titles <- paste("Simulation step:", which_steps)
  if(!is.null(extra_legend)) {
    titles <- paste(titles, paste(extra_legend, collapse = "; "), sep = "; ")
  }

  # Set up progress bar
  pb <- utils::txtProgressBar(min = 0, max = length(which_steps), style = 3)
  which_steps <- which_steps + 1
  oldpar <- NULL
  # Only store if a graphics device is open
  if(grDevices::dev.cur() != 1) {
    oldpar <- graphics::par(no.readonly = TRUE)
    on.exit({
      if(!is.null(oldpar)) {
        suppressWarnings(graphics::par(oldpar))
      }
    })
  }
  # Plotting function

  plot_simulation_step <- function(i) {
    # Create a new plot device if none exists

    sdm_st <- sdm_simul@niche_model * 0
    valuess <- sdm_simul@sdm_sim[[which_steps[i]]]
    no_cero <- .nonzero(valuess)
    sdm_st[sdm_simul@cellIDs] <- valuess
    sdm_st <- sdm_simul@niche_model + sdm_st

    # Determine colors based on values
    maxv <- raster::maxValue(sdm_st)
    minv <- raster::minValue(sdm_st)

    colores <- if(maxv == 1 && minv == 1) {
      suit_color
    } else if(maxv == 2 && minv == 2) {
      occupied_color
    } else if(maxv == 2 && minv != 0) {
      c(suit_color, occupied_color)
    } else if((maxv == 2 && nrow(no_cero) > 2) || (maxv == 2 && minv == 0)) {
      c(bg_color, suit_color, occupied_color)
    } else {
      c(bg_color, suit_color)
    }

    # Set plotting parameters

    #x11()
    raster::plot(sdm_st, main = titles[i], col = colores,
                 legend = FALSE, xaxt = 'n', yaxt = 'n')


    utils::setTxtProgressBar(pb, i)
  }
  # Generate animation based on format
  if(fmt == "GIF") {  # Prepare plot parameters

    animation::ani.options(ani.width = ani.width,
                           ani.height = ani.height,
                           ani.res = ani.res)
    animation::saveGIF({
      for(i in seq_along(which_steps)) {
        plot_simulation_step(i)
        graphics::par(xpd = TRUE)
        graphics::legend(
          "bottom",
          legend = c("Unsuitable", "Suitable", "Occupied"),
          fill = c(bg_color, suit_color, occupied_color),
          horiz = TRUE,
          inset = -0.2,
          cex = 0.75,
          bty = "n"
        )
      }
    }, interval = gif_vel, movie.name = filename)

  } else if(fmt == "HTML") {
    dir3 <- file.path(dir2, paste0("pngs_", png_keyword))
    #dir3 <- gsub("[.]", "_", dir3)
    # Prepare plot parameters
    animation::saveHTML({
      for(i in seq_along(which_steps)) {
        plot_simulation_step(i)
        graphics::par(xpd = TRUE)
        graphics::legend(
          "bottom",
          legend = c("Unsuitable", "Suitable", "Occupied"),
          fill = c(bg_color, suit_color, occupied_color),
          horiz = TRUE,
          inset = -0.2,
          cex = 0.75,
          bty = "n"
        )
      }
    }, img.name = png_keyword,
    imgdir = dir3,
    htmlfile = filename,
    ani.width = ani.width,
    ani.height = ani.height,
    interval = 0.1,
    ani.dev = function(...) { grDevices::png(res = ani.res, ...) })
  }
  return(invisible())
}
