/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.formats;

import java.math.BigInteger;

import org.docx4j.XmlUtils;
import org.docx4j.dml.CTLineProperties;
import org.docx4j.dml.CTPresetLineDashProperties;
import org.docx4j.dml.STPresetLineDashVal;
import org.docx4j.wml.CTBorder;
import org.pptx4j.jaxb.Context;

public class BorderProperties {
	private ColorProperties borderColor;
	private String borderStyle;
	private int borderWidth;

	public BorderProperties(ColorProperties borderColor, String borderStyle,
			int borderWidth) {
		super();
		this.borderColor = borderColor;
		this.borderStyle = borderStyle;
		this.borderWidth = borderWidth;
	}

	public ColorProperties getBorderColor() {
		return borderColor;
	}
	public boolean hasBorder() {
		return !borderStyle.equals("none") && borderWidth > 0;
	}
	public String getBorderStyle() {
		return borderStyle;
	}

	public int getBorderWidth() {
		return borderWidth;
	}

	public CTBorder get_docx_bd (){
		CTBorder border = new CTBorder();
		if( borderWidth > 0 && !borderStyle.equals("none") ){
			border.setColor(borderColor.getHexCode());
		    border.setSz(BigInteger.valueOf((long)( borderWidth * 4 )));
		    if( borderStyle.equals("solid") )
		    	border.setVal(org.docx4j.wml.STBorder.SINGLE);
		    else if( borderStyle.equals("dotted") )
		    	border.setVal(org.docx4j.wml.STBorder.DOTTED);
		    else if( borderStyle.equals("dashed") )
		    	border.setVal(org.docx4j.wml.STBorder.DASHED);
		    else border.setVal(org.docx4j.wml.STBorder.SINGLE);
		} else {
			border.setVal(org.docx4j.wml.STBorder.NONE);
			border.setSz(BigInteger.valueOf(0));
		}
	    return border;
	}
	
	
	public CTLineProperties get_pptx_bd (String borderSuffix) throws Exception {
		CTLineProperties border = new CTLineProperties();
		if( borderWidth > 0 && !borderStyle.equals("none") ){
			border.setSolidFill(borderColor.getSolidFill());
			border.setW(borderWidth*12700);
			CTPresetLineDashProperties lineStyle = new CTPresetLineDashProperties();
			
		    if( borderStyle.equals("solid") )
		    	lineStyle.setVal(STPresetLineDashVal.SOLID);
		    else if( borderStyle.equals("dotted") )
		    	lineStyle.setVal( STPresetLineDashVal.SYS_DOT);
		    else if( borderStyle.equals("dashed") )
		    	lineStyle.setVal( STPresetLineDashVal.SYS_DASH);
		    else lineStyle.setVal(STPresetLineDashVal.SOLID);
	    
		    border.setPrstDash(lineStyle);
		    
		} else {
			String borderStr = "<a:ln" + borderSuffix + " xmlns:a=\"http://schemas.openxmlformats.org/drawingml/2006/main\" w=\"12700\" cap=\"flat\" cmpd=\"sng\" algn=\"ctr\"><a:noFill /><a:prstDash val=\"solid\" /><a:round /><a:headEnd type=\"none\" w=\"med\" len=\"med\" /><a:tailEnd type=\"none\" w=\"med\" len=\"med\" /></a:ln" + borderSuffix + ">";
			border = ((CTLineProperties)XmlUtils.unmarshalString(borderStr, Context.jcPML, CTLineProperties.class) );
		}
	    return border;
	}
	
	public String get_html_bd (String side) throws Exception {
		String out = "";
		
		if( borderWidth < 1 ) return "";
		if( borderStyle.equals("none") ) return "";
		
		String prefix = "border-" + side + "-";
		out = out.concat(prefix);
		out = out.concat("color:");
		out = out.concat(borderColor.getrgba());
		out = out.concat(";");

		out = out.concat(prefix);
		out = out.concat("style:");
	    if( borderStyle.equals("solid") || 
	    		borderStyle.equals("dotted") || 
	    		borderStyle.equals("dashed") )
	    	out = out.concat(borderStyle);
	    else out = out.concat("solid");
		out = out.concat(";");

		out = out.concat(prefix);
		out = out.concat("width:");
		out = out + borderWidth;
		out = out.concat("px;");

		return out;
	}
	

}
