/*
 * This file is part of ReporteRs
 * Copyright (c) 2014, David Gohel All rights reserved.
 * This program is licensed under the GNU GENERAL PUBLIC LICENSE V3.
 * You may obtain a copy of the License at :
 * http://www.gnu.org/licenses/gpl.html
 */

package org.lysis.reporters.docs;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import org.apache.commons.io.IOUtils;
import org.docx4j.XmlUtils;
import org.docx4j.docProps.core.CoreProperties;
import org.docx4j.docProps.core.dc.elements.SimpleLiteral;
import org.docx4j.model.structure.SectionWrapper;
import org.docx4j.openpackaging.contenttype.ContentType;
import org.docx4j.openpackaging.exceptions.Docx4JException;
import org.docx4j.openpackaging.packages.WordprocessingMLPackage;
import org.docx4j.openpackaging.parts.DocPropsCorePart;
import org.docx4j.openpackaging.parts.PartName;
import org.docx4j.openpackaging.parts.WordprocessingML.AlternativeFormatInputPart;
import org.docx4j.openpackaging.parts.WordprocessingML.MainDocumentPart;
import org.docx4j.openpackaging.parts.WordprocessingML.NumberingDefinitionsPart;
import org.docx4j.relationships.Relationship;
import org.docx4j.wml.CTAltChunk;
import org.docx4j.wml.ContentAccessor;
import org.docx4j.wml.Numbering;
import org.docx4j.wml.P;
import org.docx4j.wml.PPr;
import org.docx4j.wml.SectPr;
import org.docx4j.wml.Styles;
import org.docx4j.wml.Tbl;
import org.lysis.reporters.docx4r.elements.PageBreak;
import org.lysis.reporters.docx4r.elements.SectionFactory;
import org.lysis.reporters.docx4r.tools.BookmarkObject;
import org.lysis.reporters.docx4r.tools.DocExplorer;
import org.lysis.reporters.formats.ParProperties;
import org.lysis.reporters.img.DrawingML;
import org.lysis.reporters.lists.NumberingDefinition;
import org.lysis.reporters.tables.FlexTable;
import org.lysis.reporters.text.CodeBlock;
import org.lysis.reporters.text.ParagraphSet;
import org.lysis.reporters.text.RScript;
import org.lysis.reporters.text.TOC;
import org.lysis.reporters.tools.Debug;


public class docx4R {
	
	// document
	private WordprocessingMLPackage basedoc;

	private NumberingDefinition numberingDefinition;
	private NumberingDefinitionsPart ndp;
	private long ordered_num_id;
	private long unordered_num_id;
	// element index in the document
	private int eltIndex;

	// stylename names and ids
	private LinkedHashMap<String, String> styleDefinitions;
	
	// What character may be used to separate styles and 
	//levels in \t switch field argument for TOC
	private String listSeparator;
	
	private SectionFactory sessionFactory;
	private SectPr current_sectionPr;
	public docx4R ( ) {
		styleDefinitions = new LinkedHashMap<String, String>();
		sessionFactory = new SectionFactory();
		eltIndex=0;
	}
	
	public void setListSeparator(String sep){
		listSeparator = sep;
	}
	public String getListSeparator(){
		return listSeparator ;
	}
	
	private void setListSeparator(){
		MainDocumentPart mdp = basedoc.getMainDocumentPart();
		String sep = mdp.getDocumentSettingsPart().getJaxbElement().getListSeparator().getVal();
		listSeparator = sep;
	}

	private void listStyleNames(){
		
		Styles styles = basedoc.getMainDocumentPart().getStyleDefinitionsPart().getJaxbElement();      
		for ( org.docx4j.wml.Style s : styles.getStyle() ) {			
			if( s.getType().equals("paragraph") ){
				try{
					styleDefinitions.put( s.getAliases().getVal(), s.getName().getVal() );
				} catch( Exception e){
					styleDefinitions.put( s.getStyleId(), s.getName().getVal() );
				}
				
			}
		}
	}

	public WordprocessingMLPackage getBaseDocument(){
		return basedoc;
	}
	public NumberingDefinitionsPart getNumberingDefinitionsPart(){
		return ndp;
	}
	public long getOrderedNumID(){
		return ordered_num_id;
	}
	public long getUnorderedNumID(){
		return unordered_num_id;
	}
	
	public String[] getStyleNames(){
		listStyleNames();
		String[] stylenames = new String[styleDefinitions.size()*2];
		int i = 0 ;
		for (Iterator<String> it1 = styleDefinitions.keySet().iterator(); it1.hasNext();) {
			stylenames[i] = it1.next();
			i++;
		}
		
		for (Map.Entry<String, String> entry : styleDefinitions.entrySet()) {
			stylenames[i] = entry.getValue();
			i++;
		}
		return stylenames;
	}

	public int[] getSectionDimensions(){
		int[] out = new int[6];

		List<SectionWrapper> sections = basedoc.getDocumentModel().getSections();
		SectPr sectPr = sections.get(sections.size() - 1).getSectPr();
		out[0] = -1;out[1] = -1;out[2] = -1;out[3] = -1;out[4] = -1;out[5] = -1;

		if (sectPr != null ){
			out[0] = sectPr.getPgSz().getW().intValue();
			out[1] = sectPr.getPgSz().getH().intValue();
			out[2] = sectPr.getPgMar().getTop().intValue();
			out[3] = sectPr.getPgMar().getRight().intValue();
			out[4] = sectPr.getPgMar().getBottom().intValue();
			out[5] = sectPr.getPgMar().getLeft().intValue();
		}
		
		return out;

	}
	public void setBaseDocument(String baseDocFileName, NumberingDefinition numbDef) throws Exception{
		try {
			basedoc = WordprocessingMLPackage.load(new FileInputStream(new File(baseDocFileName)));
		} catch (FileNotFoundException e) {
			throw new Exception ("Cannot load base document [" +  baseDocFileName + "]. File is not found.");
		} catch (Docx4JException e) {
			throw new Exception ("Cannot load base document [" +  baseDocFileName + "]. File is found but a Docx4J exception occured.");
		}
		listStyleNames();
		setListSeparator();
		sessionFactory.setPageDimensions(basedoc.getDocumentModel().getSections());
		sessionFactory.setPageBreak(false);
		current_sectionPr = sessionFactory.getOriginalSectPr();

		setNumberingDefinitionsPart(numbDef);
	}
	
	private void setNumberingDefinitionsPart(NumberingDefinition numbDef) throws Exception{
		
		if( Debug.debug_list ) System.err.println("setNumberingDefinitionsPart" );
		
		numberingDefinition =  numbDef;
		
		Numbering existingNb;
		try{
			existingNb = basedoc.getMainDocumentPart().getNumberingDefinitionsPart().getJaxbElement();
		} catch( Exception e ){
			existingNb = new Numbering();
		}
				
		numberingDefinition.stackToExistingList(existingNb);
		if( Debug.debug_list ) System.err.println("stacked " );

		Numbering newNumber = (Numbering) (numberingDefinition.get_docx_elt());

		ndp = new NumberingDefinitionsPart();
		ndp.setJaxbElement( newNumber );
		basedoc.getMainDocumentPart().addTargetPart(ndp);
		
		ordered_num_id = numberingDefinition.getOrderedNumID();
		unordered_num_id = numberingDefinition.getUnorderedNumID();
		
		restartNumbering();
	}
	
	private void setPreviousContentWithSection(SectPr sp){

    	org.docx4j.wml.P p = new P(); 
    	PPr createPPr = new PPr(); 
    	SectPr spcopy = XmlUtils.deepCopy(sp);
    	createPPr.setSectPr(spcopy); 
    	p.setPPr(createPPr); 
    	
        basedoc.getMainDocumentPart().getContent().add(p);//ICI
		basedoc.getDocumentModel().refresh();
	}

	private void setEndSection(SectPr sp){

    	PPr createPPr = new PPr(); 
    	createPPr.setSectPr(sp); 
    	
        basedoc.getMainDocumentPart().getContent().add(createPPr);//ICI
		basedoc.getDocumentModel().refresh();
	}
	
	private void eventManage(){
		sessionFactory.updateDocument(basedoc);
	}
	
	
	public void startSection(boolean landscape, int ncol, int bet_width, boolean breakpage){
		
		setPreviousContentWithSection( current_sectionPr );
		
		sessionFactory.open();
		sessionFactory.setPageBreak(breakpage);
		sessionFactory.setColumns(ncol, bet_width);
		current_sectionPr = sessionFactory.getNewSectPr(landscape); 
	}
	
	public void addColumnBreak(){
		if( !sessionFactory.isOpen() || sessionFactory.getCols() < 2 ){
			return;
		}
		sessionFactory.setColumnBreak(true);
	}
		
	public void incrementElementIndex( int inc) {
		eltIndex = eltIndex + inc;
	}
	// used by addPlot.docx to know the unique idx 
	public int getElementIndex( ) {
		if( Debug.debug) System.err.println("getElementIndex:" + eltIndex);
		return eltIndex;
	}
	
	public void writeDocxToStream(String target) throws Exception{
		if( sessionFactory.isOpen() ){
        	SectPr.Type type = new SectPr.Type();
        	type.setVal("continuous");
        	current_sectionPr.setType(type);
        	setEndSection(current_sectionPr);
		}
		
		File f = new File(target);
		try {
			basedoc.save(f);
		} catch (Docx4JException e) {
			throw new Exception ("Cannot save document [" +  target + "]. A Docx4J exception occured.");
		}
	}

	public void setDocPropertyTitle(String value){
		DocPropsCorePart docProps = basedoc.getDocPropsCorePart();
		CoreProperties cp = docProps.getJaxbElement();
		org.docx4j.docProps.core.dc.elements.ObjectFactory dcElfactory = new org.docx4j.docProps.core.dc.elements.ObjectFactory();
		SimpleLiteral literal = dcElfactory.createSimpleLiteral();
		literal.getContent().add( value );
		cp.setTitle(dcElfactory.createTitle(literal));
	}
	
	public void setDocPropertyCreator(String value){
		DocPropsCorePart docProps = basedoc.getDocPropsCorePart();
		CoreProperties cp = docProps.getJaxbElement();
		org.docx4j.docProps.core.dc.elements.ObjectFactory dcElfactory = new org.docx4j.docProps.core.dc.elements.ObjectFactory();
		SimpleLiteral literal = dcElfactory.createSimpleLiteral();
		literal.getContent().add(value);
		cp.setCreator(literal);
	}

	public String[] getWords( boolean body, boolean header, boolean footer ) throws Exception{
		return DocExplorer.getString(basedoc, body, header, footer);
	}
	public String[] getWords( String bookmark ) throws Exception{
		return DocExplorer.getString(basedoc, bookmark);
	}
	public String[] getBookMarks(boolean body, boolean header, boolean footer ) throws Exception{
		return DocExplorer.getBookmarkNames(basedoc, body, header, footer);
	}

	public void deleteBookmark( String bookmark ){
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);

		if( bo.exists() ){
			P p = bo.getP();
			((ContentAccessor)p.getParent()).getContent().remove(p);
		}
	}
	
	public void deleteBookmarkNextContent( String bookmark ){
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);

		if( bo.exists() ){
			P p = bo.getP();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    if( ((ContentAccessor)p.getParent()).getContent().size() > (i+1) )
		    	((ContentAccessor)p.getParent()).getContent().remove(i+1);
		}
	}
	
	public void addPageBreak( ) throws Exception{
		eltIndex++;
		basedoc.getMainDocumentPart().addObject( PageBreak.getBreak() );
	}
	public void add( Object obj ) throws Exception{
		eltIndex++;
		basedoc.getMainDocumentPart().addObject(obj);
	}


	public void add( TOC toc) throws Exception{

		toc.setDOCXMLPackage(basedoc);
		toc.setUID(eltIndex+1);

		basedoc.getMainDocumentPart().getContent().add(toc.get_docx_elt());

		eltIndex = (int) toc.nextUniqueId();
		eventManage();
	}
	
	public void add( FlexTable dt, ParProperties pp) throws Exception{
		dt.setParProperties(pp);
		dt.setUID(eltIndex+1);
		dt.setDOCXReference(this);

		Tbl tbl;
		tbl = dt.get_docx_elt();
		basedoc.getMainDocumentPart().addObject(tbl);
		eltIndex = (int) dt.nextUniqueId();
		eventManage();
	}
	
	public void add( FlexTable dt, ParProperties pp, String bookmark) throws Exception{
		
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);
		
		if( bo.exists() ){
			P p = bo.getP();

			dt.setParProperties(pp);
			dt.setUID(eltIndex+1);
			dt.setDOCXReference(this);
			dt.set_docx_bookmark(bookmark, bo.getBookmarkID());
			Tbl tbl = dt.get_docx_elt();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().add(i+1, tbl);
			eltIndex = (int) dt.nextUniqueId();

		} else throw new Exception("can't find bookmark '" + bookmark + "'." );

	}
	
	public void add( RScript rscript, ParProperties pp ) throws Exception{
		
		rscript.setDOCXMLPackage(basedoc);
		rscript.setParProperties(pp);
		rscript.setUID(eltIndex + 1 );
		basedoc.getMainDocumentPart().getContent().addAll(rscript.get_docx_elt());
		eltIndex = (int) rscript.nextUniqueId();
		eventManage();
	}
	
	public void add( RScript rscript, ParProperties pp, String bookmark ) throws Exception{
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);

		if( bo.exists() ){
		
			rscript.set_docx_bookmark(bookmark, bo.getBookmarkID());
			rscript.setDOCXMLPackage(basedoc);
			rscript.setParProperties(pp);
			rscript.setUID(eltIndex+1);

			P p = bo.getP();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);
		    Vector<P> pars = rscript.get_docx_elt();
		    for( int pid = 0 ; pid < rscript.size() ; pid++ ){
		    	((ContentAccessor)p.getParent()).getContent().add(i+pid, pars.get(pid));
		    }
			eltIndex = eltIndex + rscript.size() ;
			
		} else throw new Exception("can't find bookmark '" +bookmark+"'." );
	}

	public void add( CodeBlock script, ParProperties pp ) throws Exception{
		
		script.setDOCXMLPackage(basedoc);
		script.setParProperties(pp);
		script.setUID(eltIndex + 1 );
		basedoc.getMainDocumentPart().getContent().addAll(script.get_docx_elt());
		eltIndex = (int) script.nextUniqueId();
		eventManage();
	}
	
	
	public void add( CodeBlock script, ParProperties pp, String bookmark ) throws Exception{
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);

		if( bo.exists() ){
		
			script.set_docx_bookmark(bookmark, bo.getBookmarkID());
			script.setDOCXMLPackage(basedoc);
			script.setParProperties(pp);
			script.setUID(eltIndex+1);

			P p = bo.getP();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);
		    Vector<P> pars = script.get_docx_elt();
		    for( int pid = 0 ; pid < script.size() ; pid++ ){
		    	((ContentAccessor)p.getParent()).getContent().add(i+pid, pars.get(pid));
		    }
			eltIndex = eltIndex + script.size() ;
			
		} else throw new Exception("can't find bookmark '" +bookmark+"'." );
	}


	public void restartNumbering(){
		if( Debug.debug_list ) System.err.println("restartNumbering: " + ordered_num_id );
		ordered_num_id = ndp.restart(ordered_num_id, 0, 1);
		if( Debug.debug_list ) System.err.println("unordered_num_id=" + unordered_num_id);
	}

	public void addWithStyle( ParagraphSet par, String stylename ) throws Exception {
	
		P p = basedoc.getMainDocumentPart().createStyledParagraphOfText(stylename, "");

		par.setDOCXMLPackage(basedoc);
		par.setPPr(p.getPPr());
		par.setUID(eltIndex + 1 );
		
		if( par.size() > 0 ){
			Vector<P> parset = par.get_docx_elt();
			for( int i = 0 ; i < parset.size() ; i++ )
				basedoc.getMainDocumentPart().addObject(parset.get(i));
		}
		
		eltIndex = eltIndex + par.size();
		eventManage();
	}

	public void addWithStyle( ParagraphSet par, String stylename, String bookmark ) throws Exception {
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);
		if( bo.exists() ){
			
			par.setDOCXMLPackage(basedoc);
			par.setPPr(stylename);
			par.setUID(eltIndex + 1);
			par.set_docx_bookmark(bookmark, bo.getBookmarkID());

			P p = bo.getP();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);
			Vector<P> parset = par.get_docx_elt();
	    	((ContentAccessor)p.getParent()).getContent().addAll(i, parset );
			eltIndex = eltIndex + par.size();

		} else System.err.println("can't find bookmark '" +bookmark+"'.");
	}
	
	public void add( ParagraphSet par ) throws Exception {
		
//		par.getParProperties().setOrderedNumid(ordered_num_id);
//		par.getParProperties().setUnorderedNumid(unordered_num_id);
//		par.setDOCXMLPackage(basedoc);
		par.setUID(eltIndex + 1 );
		
		if( par.size() > 0 ){
			Vector<P> parset = par.get_docx_elt();
			for( int i = 0 ; i < parset.size() ; i++ )
				basedoc.getMainDocumentPart().addObject(parset.get(i));
		}
		
		eltIndex = eltIndex + par.size();
		eventManage();
	}

	public void add( ParagraphSet par, String bookmark) throws Exception {
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);
		if( bo.exists() ){
//			par.getParProperties().setOrderedNumid(ordered_num_id);
//			par.getParProperties().setUnorderedNumid(unordered_num_id);
//			par.setDOCXMLPackage(basedoc);
			par.setUID(eltIndex + 1 );
			par.set_docx_bookmark(bookmark, bo.getBookmarkID());

			P p = bo.getP();
		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);
			Vector<P> parset = par.get_docx_elt();
	    	((ContentAccessor)p.getParent()).getContent().addAll(i, parset );
			eltIndex = eltIndex + par.size();

		} else System.err.println("can't find bookmark '" +bookmark+"'.");
	}	

	public void add(org.lysis.reporters.img.Image img, ParProperties pp)
			throws Exception {

		img.setDOCXMLPackage(basedoc);
		img.setParProperties(pp);
		img.setUID(eltIndex+1);
		basedoc.getMainDocumentPart().addObject(img.get_docx_elt());
		eltIndex = (int) img.nextUniqueId();
		eventManage();
	}

	public void add ( org.lysis.reporters.img.Image img, ParProperties pp, String bookmark ) throws Exception {
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);
		if( bo.exists() ){
			P p = bo.getP();
			img.setDOCXMLPackage(basedoc);
			img.setParProperties(pp);
			img.setUID(eltIndex+1);
			img.set_docx_bookmark(bookmark, bo.getBookmarkID());

		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);
	    	img.set_docx_bookmark(bookmark, bo.getBookmarkID());
	    	((ContentAccessor)p.getParent()).getContent().add(i, 
	    			img.get_docx_elt());
			eltIndex = (int) img.nextUniqueId();
		} else System.err.println("can't find bookmark '" +bookmark+"'.");;

	}

	public void add ( DrawingML dml, ParProperties pp ) throws Exception {

		dml.setDOCXMLPackage(basedoc);
		dml.setParProperties(pp);
		dml.setUID(eltIndex+1);
		dml.setExt(dml.getWidth(), dml.getHeight());
		
		P altp = dml.get_docx_elt();			
		basedoc.getMainDocumentPart().addObject(altp);
    	eltIndex = eltIndex + 2;
    	eventManage();
	}

	public void add (DrawingML dml, ParProperties pp, String bookmark) throws Exception {
		BookmarkObject bo = DocExplorer.getBookmarkObject(bookmark, basedoc);

		if( bo.exists() ){
			P p = bo.getP();
			dml.setDOCXMLPackage(basedoc);
			dml.setParProperties(pp);
			dml.setUID(eltIndex+1);
			dml.set_docx_bookmark(bookmark, bo.getBookmarkID());
			dml.setExt(dml.getWidth(), dml.getHeight());

		    int i = ((ContentAccessor)p.getParent()).getContent().indexOf(p);
		    ((ContentAccessor)p.getParent()).getContent().remove(i);

			dml.set_docx_bookmark(bookmark, bo.getBookmarkID());
			
	    	((ContentAccessor)p.getParent()).getContent().add(i, dml.get_docx_elt());
			eltIndex = (int) dml.nextUniqueId();

		} else throw new Exception("can't find bookmark '" +bookmark+"'." );
	}
	

//    public void insertExternalDocx( String file) throws Exception {
//    		eltIndex = eltIndex+1;
//    		InputStream s2 = new FileInputStream(new File(file));
//    		String CONTENT_TYPE = "application/vnd.openxmlformats-officedocument.wordprocessingml.document";
//    		MainDocumentPart main = basedoc.getMainDocumentPart();
//			byte[] bytes = IOUtils.toByteArray(s2);
//			s2.close();
//    		AlternativeFormatInputPart afiPart = new AlternativeFormatInputPart(new PartName("/part" + 
//    			(eltIndex) + ".docx"));
//            afiPart.setContentType(new ContentType(CONTENT_TYPE));
//            afiPart.setBinaryData(bytes);
//            Relationship altChunkRel = main.addTargetPart(afiPart);
//
//            CTAltChunk chunk = new CTAltChunk();
//            chunk.setId(altChunkRel.getId());
//
//            main.addObject(chunk);
//            eltIndex++;
//    }
    public void insertExternalDocx( String file) throws Exception {

		MainDocumentPart main = basedoc.getMainDocumentPart();
		
		AlternativeFormatInputPart afiPart = new AlternativeFormatInputPart(new PartName("/doc_" + 
    			(eltIndex) + ".docx") );
		afiPart.setBinaryData(
				new FileInputStream(file) );
		
		afiPart.setContentType(new ContentType("application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml")); //docx

		Relationship altChunkRel = main.addTargetPart(afiPart);
		
		CTAltChunk ac = new CTAltChunk();
		ac.setId(altChunkRel.getId());

		main.addObject(ac);
		eltIndex++;

}
    
    public void insertExternalHTML( String file) throws Exception {
		eltIndex = eltIndex+1;
		InputStream s2 = new FileInputStream(new File(file));
		String CONTENT_TYPE = "text/html";
		MainDocumentPart main = basedoc.getMainDocumentPart();
		byte[] bytes = IOUtils.toByteArray(s2);
		AlternativeFormatInputPart afiPart = new AlternativeFormatInputPart(new PartName("/doc_" + 
			(eltIndex) + ".html"));
        afiPart.setContentType(new ContentType(CONTENT_TYPE));
        afiPart.setBinaryData(bytes);
        Relationship altChunkRel = main.addTargetPart(afiPart);

        CTAltChunk chunk = new CTAltChunk();
        chunk.setId(altChunkRel.getId());

        main.addObject(chunk);
        eltIndex++;
}
}

