#' @title Plot Kaplan-Meier survival curve.
#' @description The function `get_km_survival_curve` uses to draw the Kaplan-Meier survival curve.
#' @param km_data A data frame, including survival status, survival time, and risk score of each sample. The data frame can be generated by the function `get_risk_score`.
#' @param cut_point The threshold uses to classify patients into two subgroups with different OS.
#' @param TRAIN  Logical,if set to TRUE,the 'cut_point' is generated by the median of the risk score; Otherwise,'cut_point' can be customized.
#' @param risk.table Allowed values include:TRUE or FALSE specifying whether to show or not the risk table. Default is FALSE.
#' @importFrom survival survfit
#' @importFrom survminer ggsurvplot
#' @importFrom stats quantile
#' @importFrom survival Surv
#' @export
#' @return No return, plot the Kaplan-Meier survival curve.
#' @examples
#' #load the data.
#' data(km_data)
#' #perform the function `get_km_survival_curve`.
#' get_km_survival_curve(km_data,cut_point,TRAIN = TRUE,risk.table=TRUE)

get_km_survival_curve<-function(km_data,cut_point,TRAIN=TRUE,risk.table=TRUE){
  km_data<-as.data.frame(km_data)
  newdata<-matrix(ncol = 3,nrow=dim(km_data)[1])
  rownames(newdata)<-rownames(km_data)
  newdata[,1]<-km_data$survival
  newdata[,2]<-km_data$event
  newdata[,3]<-km_data$multiple_score
  colnames(newdata)<-c("survival","event","multiple_score")
  newdata[,3]<-as.numeric(as.matrix(newdata[,3]))
  if(TRAIN){
    cut_point<-quantile(newdata[,3],probs=0.5,na.rm =FALSE,name=FALSE,type=7,digits = 7)
    cat(cut_point)
  }else
    cut_point<-cut_point
  lo<-which(newdata[,3]> cut_point)                                                     #####训练集阈值分类
  lo1<-which(newdata[,3] <= cut_point)
  newdata[lo,3]<-"high"
  newdata[lo1,3]<-"low"
  newdata<-as.data.frame(newdata)
  newdata$survival<-as.numeric(newdata$survival)
  newdata$event<-as.numeric(newdata$event)
  fit <- survfit(Surv(survival, event) ~multiple_score, data = newdata)
  ggsurvplot(fit, data = newdata, risk.table=risk.table, conf.int = FALSE,pval = TRUE)
}
