% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/obn_ir.R
\name{ob_numerical_ir}
\alias{ob_numerical_ir}
\title{Optimal Binning using Isotonic Regression (PAVA)}
\usage{
ob_numerical_ir(
  feature,
  target,
  min_bins = 3,
  max_bins = 5,
  bin_cutoff = 0.05,
  max_n_prebins = 20,
  auto_monotonicity = TRUE,
  convergence_threshold = 1e-06,
  max_iterations = 1000
)
}
\arguments{
\item{feature}{A numeric vector representing the continuous predictor variable.
Missing values (NA) are excluded from the binning process.}

\item{target}{An integer vector of binary outcomes (0/1) corresponding to
each observation in \code{feature}. Must have the same length as \code{feature}.}

\item{min_bins}{Integer. The minimum number of bins to produce. Must be \eqn{\ge} 2.
Defaults to 3.}

\item{max_bins}{Integer. The maximum number of bins to produce. Must be \eqn{\ge}
\code{min_bins}. Defaults to 5.}

\item{bin_cutoff}{Numeric. The minimum fraction of total observations required
for a bin to be considered valid. Bins with frequency < \code{bin_cutoff}
will be merged with neighbors. Value must be in (0, 1). Defaults to 0.05.}

\item{max_n_prebins}{Integer. The number of initial quantiles to generate
during the pre-binning phase. Defaults to 20.}

\item{auto_monotonicity}{Logical. If \code{TRUE}, the algorithm automatically
determines the optimal monotonicity direction (increasing or decreasing)
based on the Pearson correlation between feature values and target.
If \code{FALSE}, defaults to increasing monotonicity.
Defaults to \code{TRUE}.}

\item{convergence_threshold}{Numeric. Reserved for future use. Currently not
actively used by the PAVA algorithm, which has guaranteed convergence.
Defaults to 1e-6.}

\item{max_iterations}{Integer. Safety limit for iterative merging operations
during pre-processing steps (e.g., rare bin merging). Defaults to 1000.}
}
\value{
A list containing the binning results:
  \itemize{
    \item \code{id}: Integer vector of bin identifiers.
    \item \code{bin}: Character vector of bin labels in interval notation.
    \item \code{woe}: Numeric vector of Weight of Evidence for each bin.
    \item \code{iv}: Numeric vector of Information Value contribution per bin.
    \item \code{count}: Integer vector of total observations per bin.
    \item \code{count_pos}: Integer vector of positive cases.
    \item \code{count_neg}: Integer vector of negative cases.
    \item \code{cutpoints}: Numeric vector of upper boundaries (excluding Inf).
    \item \code{total_iv}: The total Information Value of the binned variable.
    \item \code{monotone_increasing}: Logical indicating if the final WoE trend is increasing.
    \item \code{converged}: Logical indicating successful completion.
  }
}
\description{
Performs supervised discretization of continuous numerical variables using
Isotonic Regression (specifically the Pool Adjacent Violators Algorithm - PAVA).
This method ensures a strictly monotonic relationship between bin indices and
the empirical event rate, making it ideal for applications requiring shape
constraints like credit scoring.
}
\details{
This function implements a \strong{shape-constrained} binning approach using
\strong{Isotonic Regression}. Unlike heuristic merging strategies (ChiMerge, DP),
this method finds the optimal monotonic fit in a single pass.

\strong{Core Algorithm (PAVA):}
The Pool Adjacent Violators Algorithm (Best & Chakravarti, 1990) is used to
transform the empirical event rates of initial bins into a sequence that is
either monotonically increasing or decreasing. It works by scanning the sequence
and merging ("pooling") any adjacent pairs that violate the desired trend until
a perfect fit is achieved. This guarantees an optimal solution in \eqn{O(n)} time.

\strong{Process Flow:}
\enumerate{
  \item \strong{Pre-binning:} Creates initial bins using quantiles.
  \item \strong{Stabilization:} Merges bins below \code{bin_cutoff}.
  \item \strong{Trend Detection:} If \code{auto_monotonicity = TRUE}, calculates
        the correlation between feature midpoints and bin event rates to determine
        if the relationship should be increasing or decreasing.
  \item \strong{Shape Enforcement:} Applies PAVA to the sequence of bin event rates,
        producing a new set of rates that conform exactly to the monotonic constraint.
  \item \strong{Metric Calculation:} Derives WoE and IV from the adjusted rates.
}

\strong{Advantages:}
\itemize{
  \item \strong{Global Optimality:} PAVA finds the best fit under the monotonicity constraint.
  \item \strong{No Hyperparameters:} Unlike ChiMerge's p-value threshold, PAVA requires
        no significance level tuning for the core regression step.
  \item \strong{Robustness:} Less sensitive to arbitrary thresholds compared to greedy merging.
}
}
\examples{
# Example: Forcing a monotonic WoE trend
set.seed(123)
feature <- rnorm(500)
# Create a slightly noisy but generally increasing relationship
prob <- plogis(0.5 * feature + rnorm(500, 0, 0.3))
target <- rbinom(500, 1, prob)

result <- ob_numerical_ir(feature, target,
  min_bins = 4,
  max_bins = 6,
  auto_monotonicity = TRUE
)

print(result$bin)
print(round(result$woe, 3))
print(paste("Monotonic Increasing:", result$monotone_increasing))

}
\references{
Barlow, R. E., Bartholomew, D. J., Bremner, J. M., & Brunk, H. D. (1972).
\emph{Statistical inference under order restrictions}. John Wiley & Sons.

Best, M. J., & Chakravarti, N. (1990). Active set algorithms for isotonic
regression; A unifying framework. \emph{Mathematical Programming}, 47(1-3), 425-439.
}
\seealso{
\code{\link{ob_numerical_dp}} for greedy dynamic programming approaches.
}
