
#' read_david_data_file
#'
#' @description
#' Parses a text file output by the DAVID web application (\url{https://david.ncifcrf.gov/})
#' (see details).
#'
#' @param file A file path pointing to a DAVID "Functional Annotation Cluster" or "Functional Annotation Chart" text file.
#'
#' @param output (optional) Specifies the type of output. (default "flat") This parameter can take one of three values:
#' \describe{
#'    \item{"flat":}{ If "flat" is specified, a single data.frame containing the standard DAVID output fields is returned.
#'                  For "Functional Annnotation Cluster" data, an additional column named \code{`Cluster (ES)`} is included,
#'                  containing for each gene set, comma-separated DAVID \code{`Annotation Cluster`} assignments and in parentheses,
#'                  DAVID Enrichment Scores.}
#'    \item{"hierarchic":}{ For "hierarchic" output, a list containing a set of data.frames for each \code{`Annotation Cluster`} is
#'                  returned. This only works with "Functional Annotation Cluster" output.}
#'    \item{"GSC":}{ DAVID data sets contain nested gene sets in their \code{`Genes`} column. The gene sets can be extracted as a
#'                  list of gene set vectors by specifying this option.}
#' }
#'
#' @param redundant (optional) The "Functional Annotation Cluster" output of DAVID contains fuzzy DAVID clusters in which
#' a given gene set may be assigned to multiple clusters. As a result, some gene sets can have multiple lines in a
#' "Functional Annotation Cluster" output file, resulting in redundant data.frame rows. If this value is \code{FALSE},
#' the returned "flat" data.frame will have gene set duplicates removed and the DAVID \code{`Annotation Cluster`} identities
#' of each gene set listed as comma separated values in the \code{`Cluster (ES)`} column. If \code{TRUE} than the redundancies
#' are tolerated and replicate gene set rows are not collapsed. (default: \code{FALSE})
#'
#' @param sep (optional) Specifies the separator used in the DAVID output file. This probably does not need to be specified.
#' (default "\\t")
#'
#' @details This function parses tab-separated text files from the DAVID web application
#' (\url{https://david.ncifcrf.gov/}). Two variants of DAVID output are supported,
#' specifically the data format generated by selecting "Functional Annotation Chart" or "Functional Annotation Cluster"
#' and downloading the resulting data as a text file.
#'
#' The parser expects the following fields in the data: "Category", "Term", "Count", "%", "PValue",
#' "Genes", "List Total", "Pop Hits", "Pop Total", "Fold Enrichment", "Bonferroni", "Benjamini", and "FDR".
#'
#' To create a data.frame suitable for use with \code{\link{gsnAddPathwaysData}()}, the default options are required,
#' particularly \code{output = "flat"} and \code{redundant = FALSE}.
#'
#' @return The function returns either a data.frame containing DAVID data, a list of data.frames, or a list of gene sets.
#' (see documentation for the \code{output} parameter above).
#'
#' @export
#'
#' @seealso [gsnImportDAVID()]
#'
read_david_data_file <- function( file,
                                  output = "flat",
                                  redundant = FALSE,
                                  sep = "\t" ){
  # Output can be "flat", "hierarchic" or "GSC"
  # If redundant == TRUE, flat data may contain multiple DAVID results lines per Term due to associations with
  # multiple Annotation Clusters. If FALSE, Annotation Cluster lines are condensed and comma separated.
  lines=readLines(  con = file )

  required_fields <- c("Category", "Term", "Count", "%", "PValue",
                       "Genes", "List Total", "Pop Hits", "Pop Total",
                       "Fold Enrichment", "Bonferroni", "Benjamini", "FDR")

  numerical_fields <- c( "Count", "%", "PValue", "List Total", "Pop Hits", "Pop Total",
                         "Fold Enrichment", "Bonferroni", "Benjamini", "FDR" )

  if( stringr::str_detect( string = lines[1], "^Annotation Cluster"  ) ){
    type <- 'david.cluster'

  } else if ( stringr::str_detect( string = lines[1], "Category\tTerm" ) &&
              all( required_fields %in% stringr::str_split( string = lines[1], pattern = sep, simplify = TRUE )[1,] )
  ){

    type <- 'david.chart'
  } else {
    stop( "File type not recognized" )
  }
  # Read data
  # Save Present Values:
  Annotation.Cluster <- NULL
  Enrichment.Score <- NULL
  fields <- NULL
  fields.block <- NULL

  # Collect Annotaion Cluster and Enrichment Score by Term
  # Currently only using Cluster.Score.x.Term
  #Annotation.Cluster.x.Term <- list()
  #Enrichment.Score.x.Term <- list()
  Cluster.Score.x.Term <- list()

  annotation.cluster.match <- stringr::str_match( string = lines, "^Annotation\\sCluster\\s(\\d+)"  )
  enrichment.score.match <- stringr::str_match( string = lines, "Enrichment\\sScore:\\s([\\d\\.]+)" )
  fields.detect <- stringr::str_detect( string = lines, "Category\tTerm" )
  data.split <- stringr::str_split( string = lines, pattern = "\\t" )

  data.hl <- list() # Hierarchical list
  data.fl <- list() # Flat list

  j=1 # Flat count
  k=1 # Blockwise count

  for( i in 1:length(lines) ){
    if( lines[i] == "" ) next

    if( !is.na( annotation.cluster.match[i,2] ) ){
      Annotation.Cluster <- as.numeric(  annotation.cluster.match[i,2] )
      if( !is.na( enrichment.score.match[i,2] ) ){
        Enrichment.Score <- as.numeric( enrichment.score.match[i,2] )
      } else {
        Enrichment.Score <- NA
      }
      data.hl[[Annotation.Cluster]] <- list()
      k <- 1
    } else if ( fields.detect[i] ) {
      fields.block <- data.split[[i]]
      if( is.null( fields ) ) fields <- fields.block
    } else {
      if( type == "david.cluster" ){
        if( output == 'hierarchic' || redundant ){
          stop("DEBUG")
          dat <- c(`Annotation Cluster` = Annotation.Cluster,
                   `Enrichment Score` = Enrichment.Score,
                   structure( data.split[[i]], names = fields  ) )
          data.hl[[Annotation.Cluster]][[k]] <- dat
        } else {
          dat <- structure( data.split[[i]], names = fields  )
          Term <- dat[['Term']]
          Cluster.Score.x.Term[[Term]] <- c( Cluster.Score.x.Term[[Term]],
                                             paste0( Annotation.Cluster, "(", Enrichment.Score,")" ) )
        }
        data.fl[[j]] <- dat
        k <- k + 1
      } else {
        data.fl[[j]] <- structure( data.split[[i]], names = fields  )
      }
      j <- j + 1
    }
  }
  if( length( data.hl ) > 0 ){
    for( l in 1:length( data.hl ) ){
      # Convert data.hl from list of lists to list of data.frames
      dat.df <- as.data.frame( do.call( what = rbind, args = data.hl[[l]] ) )

      # Test if values are interpretable as numeric, either a numeric, a string containing a numeric, an NA or a ""
      for( .field in colnames( dat.df ) ){
        if( all( is.na(dat.df[[.field]]) |
                 dat.df[[.field]] == "" |
                 !is.na( suppressWarnings(as.numeric( dat.df[[.field]] ) ) ) ) )
          dat.df[[.field]] <- as.numeric(dat.df[[.field]])

      }
      data.hl[[l]] <- dat.df
    }
  }

  # Convert data.fl from list of lists to list of data.frames
  dat.fl <- as.data.frame( do.call( what = rbind, args = data.fl ) )
  if( !redundant ){
    dat.fl <- unique( dat.fl )
    if( any( duplicated( dat.fl$Term ) ) ) stop("Terms are not unique.")
    # Add a column for `Annotation Cluster`(`Enrichment Score`)
    dat.fl$`Cluster (ES)` <- sapply( X = Cluster.Score.x.Term[dat.fl$Term], FUN = function(x) paste0(x, collapse = ", ") )
  }

  # Test if values are interpretable as numeric, either a numeric, a string containing a numeric, an NA or a ""
  for( .field in colnames( dat.fl ) ){
    if( all( is.na(dat.fl[[.field]]) |
             dat.fl[[.field]] == "" |
             !is.na( suppressWarnings(as.numeric( dat.fl[[.field]] ) ) ) ) )
      dat.fl[[.field]] <- as.numeric(dat.fl[[.field]])

  }
  data.fl <- dat.fl

  if( output == "hierarchic" ){
    if( type != "david.cluster" ){
      stop("Can't output \"hierarchic\" for input type '", type, "'." )
    }
    return( data.hl )
  }
  if( output == "GSC" ){
    return( extract_david_GSC( data.fl ) )
    #return( structure( stringr::str_split( data.fl$Genes, pattern = "\\s*,\\s*" ),
    #                   names = data.fl$Term ) )
  }
  if( output == "flat" ){
    return( data.fl )
  }
  stop("Output type '", output, "' is not recognized." )
} # read_david_data_file


# extract_david_GSC <- function( data ){
#   structure( stringr::str_split( data$Genes, pattern = "\\s*,\\s*" ),
#              names = data$Term )
# }

#' extract_david_GSC
#'
#' @param data A data.frame containing DAVID pathways data (without duplicates).
#'
#' @param genes.field The name of the field containing the lists of genes for each gene set (default: \code{"Genes"}).
#'
#' @param term.field The name of the field containing the ID for each gene set (default: \code{"Term"}).
#'
#' @param del A pattern specifying the delimiter for the genes in \code{genes.field}. (default: \code{"\\\\s*,\\\\s*"})
#'
#' @return A gene set collection as a list of gene set vectors, where the gene set names correspond to Terms and
#' the vectors contain gene symbols corresponding to the genes listed in \code{genes.field}.
#'
#' @seealso
#' * \code{\link{gsnAddPathwaysData}()}
#' * \code{\link{read_david_data_file}()}
#'
#' @export
#'
#'
extract_david_GSC <- function( data, genes.field = "Genes", term.field = "Term", del = "\\s*,\\s*" ){
  structure( stringr::str_split( data[[genes.field]], pattern = del ),
             names = data[[term.field]] )
}
