#' Plot method for `summary.seroincidence.by` objects
#'
#' @param object
#' a `summary.seroincidence.by` object
#' (generated by applying the `summary()`
#' method to the output of [est_seroincidence_by()]).
#' @param type
#' [character] string indicating which type of plot to generate.
#' The implemented options are:
#' - `"scatter"`: calls [strat_ests_scatterplot()] to generate a scatterplot
#' -  `"bar"`: calls `strat_ests_barplot()` to generate a barplot
#' @inheritDotParams strat_ests_scatterplot
#' @inheritDotParams strat_ests_barplot
#'
#' @return a [ggplot2::ggplot()] object
#' @export
#' @examples
#'
#' library(dplyr)
#' library(ggplot2)
#'
#' xs_data <-
#'   sees_pop_data_pk_100
#'
#' curve <-
#'   typhoid_curves_nostrat_100 %>%
#'   filter(antigen_iso %in% c("HlyE_IgA", "HlyE_IgG"))
#'
#' noise <-
#'   example_noise_params_pk
#'
#' est2 <- est_seroincidence_by(
#'   strata = c("catchment", "ageCat"),
#'   pop_data = xs_data,
#'   sr_params = curve,
#'   noise_params = noise,
#'   curve_strata_varnames= NULL,
#'   noise_strata_varnames = NULL,
#'   antigen_isos = c("HlyE_IgG", "HlyE_IgA"),
#'   num_cores = 2 # Allow for parallel processing to decrease run time
#' )
#'
#' est2sum <- summary(est2)
#'
#' est2sum |> autoplot(
#'     type ="scatter",
#'     xvar = "ageCat",
#'     color_var = "catchment",
#'     CIs = TRUE,
#'     group_var = "catchment")
#'
#' est2sum |> autoplot(
#'     type = "bar",
#'     yvar = "ageCat",
#'     color_var = "catchment",
#'     CIs = TRUE)
#'
autoplot.summary.seroincidence.by <- function(
    object,
    type,
    ...) {

  if (type == "scatter") {
    plot1 <- strat_ests_scatterplot(object, ...)

  } else if (type == "bar") {
    plot1 <- strat_ests_barplot(object,  ...)

  } else {
    cli::cli_abort(
      c(
        "Invalid plot `type` specified: {.str {type}}.",
        "i" = "Please choose either 'scatter' or 'bar'."
      )
    )
  }

  return(plot1)
}
