/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5ACmodule.h" 

#include "H5private.h"   
#include "H5ACpkg.h"     
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5MFprivate.h" 
#include "H5SLprivate.h" 

static herr_t H5AC__proxy_entry_image_len(const void *thing, size_t *image_len);
static herr_t H5AC__proxy_entry_serialize(const H5F_t *f, void *image_ptr, size_t len, void *thing);
static herr_t H5AC__proxy_entry_notify(H5AC_notify_action_t action, void *thing);
static herr_t H5AC__proxy_entry_free_icr(void *thing);

const H5AC_class_t H5AC_PROXY_ENTRY[1] = {{
    H5AC_PROXY_ENTRY_ID,         
    "Proxy entry",               
    H5FD_MEM_SUPER,              
    0,                           
    NULL,                        
    NULL,                        
    NULL,                        
    NULL,                        
    H5AC__proxy_entry_image_len, 
    NULL,                        
    H5AC__proxy_entry_serialize, 
    H5AC__proxy_entry_notify,    
    H5AC__proxy_entry_free_icr,  
    NULL,                        
}};

H5FL_DEFINE_STATIC(H5AC_proxy_entry_t);

H5AC_proxy_entry_t *
H5AC_proxy_entry_create(void)
{
    H5AC_proxy_entry_t *pentry    = NULL; 
    H5AC_proxy_entry_t *ret_value = NULL; 

    FUNC_ENTER_NOAPI(NULL)

    
    if (NULL == (pentry = H5FL_CALLOC(H5AC_proxy_entry_t)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, NULL, "can't allocate proxy entry");

    
    pentry->addr = HADDR_UNDEF;

    
    ret_value = pentry;

done:
    
    if (!ret_value)
        if (pentry)
            pentry = H5FL_FREE(H5AC_proxy_entry_t, pentry);

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_proxy_entry_add_parent(H5AC_proxy_entry_t *pentry, void *_parent)
{
    H5AC_info_t *parent    = (H5AC_info_t *)_parent; 
    herr_t       ret_value = SUCCEED;                

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(parent);
    assert(pentry);

    
    if (NULL == pentry->parents)
        if (NULL == (pentry->parents = H5SL_create(H5SL_TYPE_HADDR, NULL)))
            HGOTO_ERROR(H5E_CACHE, H5E_CANTCREATE, FAIL,
                        "unable to create skip list for parents of proxy entry");

    
    if (H5SL_insert(pentry->parents, parent, &parent->addr) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTINSERT, FAIL, "unable to insert parent into proxy's skip list");

    
    if (pentry->nchildren > 0) {
        
        assert(H5_addr_defined(pentry->addr));

        if (H5AC_create_flush_dependency(parent, pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "unable to set flush dependency on proxy entry");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_proxy_entry_remove_parent(H5AC_proxy_entry_t *pentry, void *_parent)
{
    H5AC_info_t *parent = (H5AC_info_t *)_parent; 
    H5AC_info_t *rem_parent;                      
    herr_t       ret_value = SUCCEED;             

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pentry);
    assert(pentry->parents);
    assert(parent);

    
    if (NULL == (rem_parent = (H5AC_info_t *)H5SL_remove(pentry->parents, &parent->addr)))
        HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "unable to remove proxy entry parent from skip list");
    if (!H5_addr_eq(rem_parent->addr, parent->addr))
        HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "removed proxy entry parent not the same as real parent");

    
    if (0 == H5SL_count(pentry->parents)) {
        
        assert(0 == pentry->nchildren);

        if (H5SL_close(pentry->parents) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CLOSEERROR, FAIL, "can't close proxy parent skip list");
        pentry->parents = NULL;
    } 

    
    if (pentry->nchildren > 0)
        if (H5AC_destroy_flush_dependency(parent, pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL,
                        "unable to remove flush dependency on proxy entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5AC__proxy_entry_add_child_cb(void *_item, void H5_ATTR_UNUSED *_key, void *_udata)
{
    H5AC_info_t        *parent    = (H5AC_info_t *)_item;         
    H5AC_proxy_entry_t *pentry    = (H5AC_proxy_entry_t *)_udata; 
    int                 ret_value = H5_ITER_CONT;                 

    FUNC_ENTER_PACKAGE

    
    if (H5AC_create_flush_dependency(parent, pentry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, H5_ITER_ERROR,
                    "unable to set flush dependency for virtual entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_proxy_entry_add_child(H5AC_proxy_entry_t *pentry, H5F_t *f, void *child)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pentry);
    assert(child);

    
    if (0 == pentry->nchildren) {
        
        if (!H5_addr_defined(pentry->addr))
            if (HADDR_UNDEF == (pentry->addr = H5MF_alloc_tmp(f, 1)))
                HGOTO_ERROR(H5E_CACHE, H5E_CANTALLOC, FAIL,
                            "temporary file space allocation failed for proxy entry");

        
        if (H5AC_insert_entry(f, H5AC_PROXY_ENTRY, pentry->addr, pentry, H5AC__PIN_ENTRY_FLAG) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTINSERT, FAIL, "unable to cache proxy entry");

        
        if (H5AC_mark_entry_clean(pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTCLEAN, FAIL, "can't mark proxy entry clean");

        
        if (H5AC_mark_entry_serialized(pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "can't mark proxy entry clean");

        
        if (pentry->parents)
            if (H5SL_iterate(pentry->parents, H5AC__proxy_entry_add_child_cb, pentry) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "can't visit parents");
    } 

    
    if (H5AC_create_flush_dependency(pentry, child) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTDEPEND, FAIL, "unable to set flush dependency on proxy entry");

    
    pentry->nchildren++;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static int
H5AC__proxy_entry_remove_child_cb(void *_item, void H5_ATTR_UNUSED *_key, void *_udata)
{
    H5AC_info_t        *parent    = (H5AC_info_t *)_item;         
    H5AC_proxy_entry_t *pentry    = (H5AC_proxy_entry_t *)_udata; 
    int                 ret_value = H5_ITER_CONT;                 

    FUNC_ENTER_PACKAGE

    
    if (H5AC_destroy_flush_dependency(parent, pentry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, H5_ITER_ERROR,
                    "unable to remove flush dependency for proxy entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_proxy_entry_remove_child(H5AC_proxy_entry_t *pentry, void *child)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pentry);
    assert(child);

    
    if (H5AC_destroy_flush_dependency(pentry, child) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTUNDEPEND, FAIL, "unable to remove flush dependency on proxy entry");

    
    pentry->nchildren--;

    
    if (0 == pentry->nchildren) {
        
        if (pentry->parents)
            
            if (H5SL_iterate(pentry->parents, H5AC__proxy_entry_remove_child_cb, pentry) < 0)
                HGOTO_ERROR(H5E_CACHE, H5E_BADITER, FAIL, "can't visit parents");

        
        if (H5AC_unpin_entry(pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTUNPIN, FAIL, "can't unpin proxy entry");

        
        if (H5AC_remove_entry(pentry) < 0)
            HGOTO_ERROR(H5E_CACHE, H5E_CANTREMOVE, FAIL, "unable to remove proxy entry");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5AC_proxy_entry_dest(H5AC_proxy_entry_t *pentry)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(pentry);
    assert(NULL == pentry->parents);
    assert(0 == pentry->nchildren);
    assert(0 == pentry->ndirty_children);
    assert(0 == pentry->nunser_children);

    
    pentry = H5FL_FREE(H5AC_proxy_entry_t, pentry);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5AC__proxy_entry_image_len(const void H5_ATTR_UNUSED *thing, size_t *image_len)
{
    FUNC_ENTER_PACKAGE_NOERR

    
    assert(image_len);

    
    *image_len = 1;

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

static herr_t
H5AC__proxy_entry_serialize(const H5F_t H5_ATTR_UNUSED *f, void H5_ATTR_UNUSED *image,
                            size_t H5_ATTR_UNUSED len, void H5_ATTR_UNUSED *thing)
{
    FUNC_ENTER_PACKAGE_NOERR 

        
        assert(0 && "Invalid callback?!?");

    HERROR(H5E_CACHE, H5E_CANTSERIALIZE, "called unreachable fcn.");

    FUNC_LEAVE_NOAPI(FAIL)
} 

static herr_t
H5AC__proxy_entry_notify(H5AC_notify_action_t action, void *_thing)
{
    H5AC_proxy_entry_t *pentry    = (H5AC_proxy_entry_t *)_thing;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    assert(pentry);

    switch (action) {
        case H5AC_NOTIFY_ACTION_AFTER_INSERT:
            break;

        case H5AC_NOTIFY_ACTION_AFTER_LOAD:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "invalid notify action from metadata cache");
#else  
            assert(0 && "Invalid action?!?");
#endif 
            break;

        case H5AC_NOTIFY_ACTION_AFTER_FLUSH:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "invalid notify action from metadata cache");
#else  
            assert(0 && "Invalid action?!?");
#endif 
            break;

        case H5AC_NOTIFY_ACTION_BEFORE_EVICT:
            
            assert(0 == pentry->ndirty_children);
            assert(0 == pentry->nunser_children);

            
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_DIRTIED:
            
            assert(pentry->ndirty_children > 0);

            
            break;

        case H5AC_NOTIFY_ACTION_ENTRY_CLEANED:
            
            assert(0 == pentry->ndirty_children);

            
            break;

        case H5AC_NOTIFY_ACTION_CHILD_DIRTIED:
            
            pentry->ndirty_children++;

            
            if (1 == pentry->ndirty_children)
                if (H5AC_mark_entry_dirty(pentry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTDIRTY, FAIL, "can't mark proxy entry dirty");
            break;

        case H5AC_NOTIFY_ACTION_CHILD_CLEANED:
            
            assert(pentry->ndirty_children > 0);

            
            pentry->ndirty_children--;

            
            if (0 == pentry->ndirty_children)
                if (H5AC_mark_entry_clean(pentry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTCLEAN, FAIL, "can't mark proxy entry clean");
            break;

        case H5AC_NOTIFY_ACTION_CHILD_UNSERIALIZED:
            
            pentry->nunser_children++;

            
            if (1 == pentry->nunser_children)
                if (H5AC_mark_entry_unserialized(pentry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTUNSERIALIZE, FAIL, "can't mark proxy entry unserialized");
            break;

        case H5AC_NOTIFY_ACTION_CHILD_SERIALIZED:
            
            assert(pentry->nunser_children > 0);

            
            pentry->nunser_children--;

            
            if (0 == pentry->nunser_children)
                if (H5AC_mark_entry_serialized(pentry) < 0)
                    HGOTO_ERROR(H5E_CACHE, H5E_CANTSERIALIZE, FAIL, "can't mark proxy entry serialized");
            break;

        default:
#ifdef NDEBUG
            HGOTO_ERROR(H5E_CACHE, H5E_BADVALUE, FAIL, "unknown notify action from metadata cache");
#else  
            assert(0 && "Unknown action?!?");
#endif 
    }  

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

static herr_t
H5AC__proxy_entry_free_icr(void *_thing)
{
    H5AC_proxy_entry_t *pentry    = (H5AC_proxy_entry_t *)_thing;
    herr_t              ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    if (H5AC_proxy_entry_dest(pentry) < 0)
        HGOTO_ERROR(H5E_CACHE, H5E_CANTFREE, FAIL, "unable to destroy proxy entry");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
